import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { FaUserGraduate, FaUniversity, FaGraduationCap, FaStickyNote, FaTimes, FaSpinner, FaLanguage, FaLinkedin, FaGithub, FaInfoCircle } from 'react-icons/fa';
import apiClient from '../apiClient';

interface Skill {
  id: string;
  name: string;
  level?: string;
}

interface Education {
  id: string;
  school: string;
  degree?: string;
  field?: string;
  start_year?: string;
  end_year?: string;
  description?: string;
}

interface Profile {
  full_name: string;
  university: string;
  photo?: string;
  summary?: string;
  linkedin?: string;
  github?: string;
  skills: Skill[];
  educations: Education[];
  languages: string[];
  user_id: string;
}

const ScanProfile: React.FC = () => {
  const { uuid } = useParams<{ uuid: string }>();
  const navigate = useNavigate();

  const [showModal, setShowModal] = useState(true);
  const [privateNote, setPrivateNote] = useState('');
  const [saving, setSaving] = useState(false);
  const [saveSuccess, setSaveSuccess] = useState(false);
  const [saveError, setSaveError] = useState('');
  const [loading, setLoading] = useState(true);
  const [profile, setProfile] = useState<Profile | null>(null);
  const [fetchError, setFetchError] = useState('');

  useEffect(() => {
    setShowModal(true);
    setLoading(true);
    setProfile(null);
    setFetchError('');
    if (!uuid) return;
    apiClient.get(`/api/p/${uuid}`)
      .then(res => {
        setProfile(res.data.student);
      })
      .catch(err => {
        const error: any = err;
        if (error.response && error.response.data && error.response.data.message) {
          setFetchError(error.response.data.message);
        } else {
          setFetchError('Erreur lors du chargement du profil.');
        }
      })
      .finally(() => setLoading(false));
  }, [uuid]);

  const handleSaveShortlist = async () => {
    if (!profile || !profile.user_id) return;
    setSaving(true);
    setSaveError('');
    setSaveSuccess(false);
    try {
      await apiClient.post('/api/shortlist', {
        student_id: profile.user_id,
        private_note: privateNote,
      });
      setSaveSuccess(true);
    } catch (err) {
      const error: any = err;
      if (error.response && error.response.data && error.response.data.message) {
        setSaveError(error.response.data.message);
      } else {
        setSaveError('Erreur lors de la sauvegarde.');
      }
    } finally {
      setSaving(false);
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50 flex items-center justify-center py-12 px-4">
      {/* Modal Overlay */}
      {showModal && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-40">
          <div className="bg-white rounded-2xl shadow-2xl max-w-md w-full sm:max-w-lg md:max-w-xl p-4 sm:p-8 relative animate-fadeIn max-h-[80vh] overflow-y-auto scrollbar-thin scrollbar-thumb-indigo-300 scrollbar-track-indigo-50">
            <button
              className="absolute top-4 right-4 text-gray-400 hover:text-gray-700 text-xl"
              onClick={() => navigate('/recruiter-dashboard')}
              aria-label="Close"
            >
              <FaTimes />
            </button>
            {loading ? (
              <div className="flex flex-col items-center justify-center py-12">
                <FaSpinner className="animate-spin text-3xl text-indigo-600 mb-4" />
                <p className="text-gray-600">Loading profile...</p>
              </div>
            ) : fetchError ? (
              <div className="flex flex-col items-center justify-center py-12">
                <p className="text-red-600 text-center">{fetchError}</p>
              </div>
            ) : profile ? (
              <>
                <div className="flex flex-col items-center mb-6">
                  <div className="h-20 w-20 bg-gradient-to-br from-blue-600 to-purple-600 rounded-full flex items-center justify-center mb-4 overflow-hidden">
                    {profile.photo ? (
                      <img src={profile.photo.startsWith('http') ? profile.photo : `http://127.0.0.1:8000${profile.photo}`} alt="Student photo" className="h-full w-full object-cover rounded-full" />
                    ) : (
                      <FaUserGraduate className="text-white text-3xl" />
                    )}
                  </div>
                  <h2 className="text-2xl font-bold text-gray-900 mb-1">{profile.full_name}</h2>
                  <div className="flex items-center text-gray-600 mb-1">
                    <FaUniversity className="mr-2" />
                    <span>{profile.university}</span>
                  </div>
                  {profile.educations && profile.educations.length > 0 && (
                    <div className="flex items-center text-gray-600">
                      <FaGraduationCap className="mr-2" />
                      <span>{profile.educations[0].degree} {profile.educations[0].field && `- ${profile.educations[0].field}`}</span>
                    </div>
                  )}
                  {profile.summary && (
                    <div className="w-full mt-4 text-center">
                      <h3 className="font-semibold text-gray-800 mb-1 flex items-center justify-center"><FaInfoCircle className="mr-2" />About</h3>
                      <p className="text-gray-700 text-sm">{profile.summary}</p>
                    </div>
                  )}
                  <div className="flex flex-wrap gap-4 mt-4 justify-center">
                    {profile.linkedin && (
                      <a href={profile.linkedin.startsWith('http') ? profile.linkedin : `https://www.linkedin.com/in/${profile.linkedin}`} target="_blank" rel="noopener noreferrer" className="flex items-center text-blue-700 hover:underline"><FaLinkedin className="mr-1" />LinkedIn</a>
                    )}
                    {profile.github && (
                      <a href={profile.github.startsWith('http') ? profile.github : `https://www.github.com/${profile.github}`} target="_blank" rel="noopener noreferrer" className="flex items-center text-gray-800 hover:underline"><FaGithub className="mr-1" />GitHub</a>
                    )}
                  </div>
                </div>
                {profile.skills && profile.skills.length > 0 && (
                  <div className="mb-4">
                    <span className="font-medium text-gray-700">Top 5 skills:</span>
                    <div className="flex flex-wrap gap-2 mt-2">
                      {profile.skills.slice(0, 5).map((skill: Skill, idx: number) => (
                        <span key={skill.id || idx} className="px-2 py-1 bg-indigo-100 text-indigo-800 text-xs rounded-full">
                          {skill.name} {skill.level && `(${skill.level})`}
                        </span>
                      ))}
                    </div>
                  </div>
                )}
                {profile.educations && profile.educations.length > 0 && (
                  <div className="mb-4">
                    <span className="font-medium text-gray-700">Education:</span>
                    <div className="mt-2 space-y-2">
                      {profile.educations.map((education: Education, idx: number) => (
                        <div key={education.id || idx} className="text-sm text-gray-600">
                          <div className="font-medium">{education.school}</div>
                          {education.degree && <div>{education.degree} {education.field && `- ${education.field}`}</div>}
                          {education.start_year && education.end_year && (
                            <div className="text-xs text-gray-500">{education.start_year} - {education.end_year}</div>
                          )}
                        </div>
                      ))}
                    </div>
                  </div>
                )}
                {profile.languages && profile.languages.length > 0 && (
                  <div className="mb-4">
                    <span className="font-medium text-gray-700 flex items-center"><FaLanguage className="mr-2" />Languages:</span>
                    <div className="flex flex-wrap gap-2 mt-2">
                      {profile.languages.map((lang: string, idx: number) => (
                        <span key={idx} className="px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full">
                          {lang.trim()}
                        </span>
                      ))}
                    </div>
                  </div>
                )}
                <div className="mb-4">
                  <label className="block text-gray-700 font-medium mb-1 flex items-center">
                    <FaStickyNote className="mr-2" /> Add a private note
                  </label>
                  <textarea
                    className="w-full border border-gray-300 rounded-lg p-2 text-sm focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none"
                    rows={2}
                    placeholder="Add a private note..."
                    value={privateNote}
                    onChange={e => setPrivateNote(e.target.value)}
                  />
                </div>
                <button
                  className="w-full bg-gradient-to-r from-indigo-600 to-purple-600 text-white font-semibold py-3 rounded-lg shadow-lg hover:from-indigo-700 hover:to-purple-700 transition-all duration-200 disabled:opacity-50 mb-2"
                  onClick={handleSaveShortlist}
                  disabled={saving || saveSuccess}
                >
                  {saving ? 'Saving...' : saveSuccess ? 'Added to Shortlist!' : 'Save to Shortlist'}
                </button>
                {saveError && <p className="text-sm text-red-600 text-center mt-2">{saveError}</p>}
              </>
            ) : null}
          </div>
        </div>
      )}
      {/* Optionally, background content can be blurred or dimmed */}
      <div className="max-w-lg w-full bg-white rounded-2xl shadow-xl p-8 opacity-30 pointer-events-none select-none">
        {/* Background content, not interactive when modal is open */}
      </div>
    </div>
  );
};

export default ScanProfile; 