import React, { useState, useRef, useEffect } from 'react';
import axios from 'axios';
import { useNavigate, Link } from 'react-router-dom';
import apiClient from '../apiClient';
import { 
  FaUser, FaEnvelope, FaLock, FaBuilding, FaUniversity, FaGraduationCap, 
  FaPhone, FaLinkedin, FaGithub, FaFileUpload, FaSpinner, FaPlus, FaEdit, 
  FaTrash, FaTimes, FaCheckCircle 
} from 'react-icons/fa';

interface Experience {
  company: string;
  title: string;
  start_date: string;
  end_date?: string;
  description?: string;
}

interface Skill {
  name: string;
  level: string;
}

interface Education {
  school: string;
  degree?: string;
  field?: string;
  start_year?: string;
  end_year?: string;
  description?: string;
}

const Register: React.FC = () => {
  const [formData, setFormData] = useState({
    fullName: '',
    email: '',
    phone: '',
    linkedin: '',
    github: '',
    summary: '',
    university: '',
    degree: '',
    graduationYear: '',
    skills: '',
    languages: '',
    companyName: '',
    companyDescription: '',
    companyIndustry: '',
    companySize: '',
    companyWebsite: '',
    position: '',
    department: '',
    password: '',
    confirmPassword: '',
    role: 'student',
  });

  const [experiences, setExperiences] = useState<Experience[]>([]);
  const [skillsList, setSkillsList] = useState<Skill[]>([]);
  const [educationsList, setEducationsList] = useState<Education[]>([]);
  const [showExperienceModal, setShowExperienceModal] = useState(false);
  const [showSkillModal, setShowSkillModal] = useState(false);
  const [showEducationModal, setShowEducationModal] = useState(false);
  const [editingExperience, setEditingExperience] = useState<Experience | null>(null);
  const [editingSkill, setEditingSkill] = useState<Skill | null>(null);
  const [editingEducation, setEditingEducation] = useState<Education | null>(null);
  const [experienceForm, setExperienceForm] = useState<Experience>({
    company: '',
    title: '',
    start_date: '',
    end_date: '',
    description: ''
  });
  const [skillForm, setSkillForm] = useState<Skill>({
    name: '',
    level: 'Beginner'
  });
  const [educationForm, setEducationForm] = useState<Education>({
    school: '',
    degree: '',
    field: '',
    start_year: '',
    end_year: '',
    description: ''
  });

  const [errors, setErrors] = useState<{ [key: string]: string }>({});
  const [cvLoading, setCvLoading] = useState(false);
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();
  const fileInputRef = useRef<HTMLInputElement>(null);
  const [checkingEmail, setCheckingEmail] = useState(false);

  useEffect(() => {
    const storedRole = localStorage.getItem('register_role');
    if (storedRole === 'student' || storedRole === 'recruiter') {
      setFormData((prev) => ({ ...prev, role: storedRole }));
    }
  }, []);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
    
    if (errors[name]) {
      setErrors(prev => ({ ...prev, [name]: '' }));
    }
  };

  // Experience management functions
  const openExperienceModal = (experience?: Experience) => {
    if (experience) {
      setEditingExperience(experience);
      setExperienceForm({ ...experience });
    } else {
      setEditingExperience(null);
      setExperienceForm({
        company: '',
        title: '',
        start_date: '',
        end_date: '',
        description: ''
      });
    }
    setShowExperienceModal(true);
  };

  const handleSaveExperience = () => {
    if (!experienceForm.company || !experienceForm.title || !experienceForm.start_date) {
      return;
    }

    if (editingExperience) {
      // Update existing experience
      setExperiences(prev => prev.map(exp => 
        exp === editingExperience ? { ...experienceForm } : exp
      ));
    } else {
      // Add new experience
      setExperiences(prev => [...prev, { ...experienceForm }]);
    }

    setShowExperienceModal(false);
    setEditingExperience(null);
  };

  const handleDeleteExperience = (experience: Experience) => {
    setExperiences(prev => prev.filter(exp => exp !== experience));
  };

  // Skill management functions
  const openSkillModal = (skill?: Skill) => {
    if (skill) {
      setEditingSkill(skill);
      setSkillForm({ ...skill });
    } else {
      setEditingSkill(null);
      setSkillForm({
        name: '',
        level: 'Beginner'
      });
    }
    setShowSkillModal(true);
  };

  const handleSaveSkill = () => {
    if (!skillForm.name) {
      return;
    }

    if (editingSkill) {
      // Update existing skill
      setSkillsList(prev => prev.map(skill => 
        skill === editingSkill ? { ...skillForm } : skill
      ));
    } else {
      // Add new skill
      setSkillsList(prev => [...prev, { ...skillForm }]);
    }

    setShowSkillModal(false);
    setEditingSkill(null);
  };

  const handleDeleteSkill = (skill: Skill) => {
    setSkillsList(prev => prev.filter(s => s !== skill));
  };

  // Education management functions
  const openEducationModal = (education?: Education) => {
    if (education) {
      setEditingEducation(education);
      setEducationForm({ ...education });
    } else {
      setEditingEducation(null);
      setEducationForm({
        school: '',
        degree: '',
        field: '',
        start_year: '',
        end_year: '',
        description: ''
      });
    }
    setShowEducationModal(true);
  };

  const handleSaveEducation = () => {
    if (!educationForm.school) {
      return;
    }

    if (editingEducation) {
      // Update existing education
      setEducationsList(prev => prev.map(edu => 
        edu === editingEducation ? { ...educationForm } : edu
      ));
    } else {
      // Add new education
      setEducationsList(prev => [...prev, { ...educationForm }]);
    }

    setShowEducationModal(false);
    setEditingEducation(null);
  };

  const handleDeleteEducation = (education: Education) => {
    setEducationsList(prev => prev.filter(edu => edu !== education));
  };

  const validateForm = () => {
    const newErrors: { [key: string]: string } = {};

    if (!formData.fullName.trim()) {
      newErrors.fullName = 'Full name is required';
    }

    if (!formData.email.trim()) {
      newErrors.email = 'Email is required';
    } else if (!/\S+@\S+\.\S+/.test(formData.email)) {
      newErrors.email = 'Please enter a valid email address';
    }

    if (!formData.password) {
      newErrors.password = 'Password is required';
    } else if (formData.password.length < 8) {
      newErrors.password = 'Password must be at least 8 characters';
    }

    if (formData.password !== formData.confirmPassword) {
      newErrors.confirmPassword = 'Passwords do not match';
    }

    if (formData.role === 'student') {
      if (!formData.university.trim()) {
        newErrors.university = 'University is required';
      }
      if (!formData.degree.trim()) {
        newErrors.degree = 'Degree is required';
      }
    }

    if (formData.role === 'recruiter') {
      if (!formData.companyName.trim()) {
        newErrors.companyName = 'Company name is required';
      }
      if (!formData.position.trim()) {
        newErrors.position = 'Position is required';
      }
      if (!formData.department.trim()) {
        newErrors.department = 'Department is required';
      }
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setErrors({});

    if (!validateForm()) return;

    setLoading(true);

    try {
      const requestData: Record<string, unknown> = {
        email: formData.email,
        password: formData.password,
        password_confirmation: formData.confirmPassword,
        role: formData.role,
        full_name: formData.fullName,
      };

      // Add role-specific fields
      if (formData.role === 'student') {
        Object.assign(requestData, {
          phone: formData.phone || null,
          linkedin: formData.linkedin || null,
          github: formData.github || null,
          summary: formData.summary || null,
          university: formData.university || null,
          degree: formData.degree || null,
          graduation_year: formData.graduationYear || null,
          skills: formData.skills || null,
          languages: formData.languages || null,
        });
      } else if (formData.role === 'recruiter') {
        Object.assign(requestData, {
          company_name: formData.companyName,
          company_description: formData.companyDescription || null,
          company_industry: formData.companyIndustry || null,
          company_size: formData.companySize || null,
          company_website: formData.companyWebsite || null,
          position: formData.position,
          department: formData.department,
        });
      }

      const response = await apiClient.post('/api/register', requestData);

      if (response.data.status === 'success') {
        const userId = response.data.data.user.id;
        const token = response.data.data.token;
        localStorage.setItem('token', token);
        localStorage.setItem('user', JSON.stringify(response.data.data.user));
        
        // Save experiences and skills if user is a student
        if (response.data.data.user.role === 'student') {
          try {
            // Create headers with the new token for batch requests
            const authHeaders = { headers: { Authorization: `Bearer ${token}` } };
            
            // Helper function to normalize dates (convert "2024-07" to "2024-07-01")
            const normalizeDate = (dateStr: string) => {
              if (!dateStr) return dateStr;
              // If it's already a full date (YYYY-MM-DD), return as is
              if (dateStr.match(/^\d{4}-\d{2}-\d{2}$/)) return dateStr;
              // If it's a partial date (YYYY-MM), add day
              if (dateStr.match(/^\d{4}-\d{2}$/)) return `${dateStr}-01`;
              // If it's just a year (YYYY), add month and day
              if (dateStr.match(/^\d{4}$/)) return `${dateStr}-01-01`;
              // Return as is for other formats
              return dateStr;
            };
            
            // Batch save experiences
            if (experiences.length > 0) {
              // Normalize dates in experiences before saving
              const normalizedExperiences = experiences.map(exp => ({
                ...exp,
                start_date: normalizeDate(exp.start_date),
                end_date: exp.end_date ? normalizeDate(exp.end_date) : exp.end_date
              }));
              console.log('Saving experiences:', normalizedExperiences);
              await apiClient.post(`/api/students/${userId}/experiences/batch`, {
                experiences: normalizedExperiences
              }, authHeaders);
            }
            
            // Batch save skills
            if (skillsList.length > 0) {
              console.log('Saving skills:', skillsList);
              await apiClient.post(`/api/students/${userId}/skills/batch`, {
                skills: skillsList
              }, authHeaders);
            }

            // Batch save educations
            if (educationsList.length > 0) {
              console.log('Saving educations:', educationsList);
              await apiClient.post(`/api/students/${userId}/educations/batch`, {
                educations: educationsList
              }, authHeaders);
            }
            
            console.log('All batch saves completed successfully');
          } catch (batchError) {
            console.error('Error saving experiences/skills/educations:', batchError);
            if (axios.isAxiosError(batchError)) {
              console.error('Batch error details:', {
                status: batchError.response?.status,
                statusText: batchError.response?.statusText,
                data: batchError.response?.data,
                url: batchError.config?.url,
                method: batchError.config?.method
              });
            }
            // Show user-friendly error but still allow navigation
            setErrors({ general: 'Profile created successfully, but some experiences/skills/educations could not be saved. You can add them later in your profile.' });
            // Show error but still allow navigation
            setErrors({ general: 'Profile created successfully, but some experiences/skills/educations could not be saved. You can add them later in your profile.' });
          }
          
          navigate('/student-dashboard');
        } else if (response.data.data.user.role === 'recruiter') {
          navigate('/recruiter-dashboard');
        } else {
          navigate('/dashboard');
        }
      } else {
        setErrors({ general: response.data.message || 'Registration failed' });
      }
    } catch (err) {
      if (axios.isAxiosError(err)) {
        if (err.response?.status === 422) {
          const validationErrors = err.response.data.errors;
          if (validationErrors) {
            const newErrors: { [key: string]: string } = {};
            Object.keys(validationErrors).forEach(key => {
              newErrors[key] = validationErrors[key][0];
            });
            setErrors(newErrors);
          } else {
            setErrors({ general: err.response.data.message || 'Validation failed' });
          }
        } else {
          setErrors({ general: err.response?.data?.message || 'An error occurred' });
        }
      } else {
        setErrors({ general: 'An unexpected error occurred' });
      }
    } finally {
      setLoading(false);
    }
  };

  const handleCVUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      setCvLoading(true);
      setErrors({});
      const file = e.target.files[0];
      console.log('Uploading CV file:', {
        name: file.name,
        size: file.size,
        type: file.type
      });
      
      const cvFormData = new FormData();
      cvFormData.append('cv', file);
      
      console.log('FormData created, making API call to /api/cv/analyze...');
      
      try {
        const response = await apiClient.post('/api/cv/analyze', cvFormData, {
          headers: { 'Content-Type': 'multipart/form-data' },
        });
        
        console.log('CV Analysis API Response Status:', response.status);
        console.log('CV Analysis Response Headers:', response.headers);
        console.log('CV Analysis Response Data:', response.data);
        console.log('Response data type:', typeof response.data);
        console.log('Response data keys:', Object.keys(response.data || {}));
        
        // Check if response has the expected structure
        if (response.data && typeof response.data === 'object') {
          console.log('Response structure:');
          console.log('- structured_cv:', typeof response.data.structured_cv, response.data.structured_cv);
          console.log('- llm_raw:', typeof response.data.llm_raw, response.data.llm_raw?.substring(0, 100) + '...');
          console.log('- extracted_text:', typeof response.data.extracted_text, response.data.extracted_text?.substring(0, 100) + '...');
        }
        
        // Check if we have structured data
        if (response.data.structured_cv && typeof response.data.structured_cv === 'object') {
          const parsedData = response.data.structured_cv;
          console.log('✅ Using structured CV data:', parsedData);
          console.log('Available keys in parsed data:', Object.keys(parsedData));
          
          // Fill form with parsed data
          fillFormWithParsedData(parsedData);
        } else if (response.data.llm_raw) {
          console.log('⚠️ Structured CV not available, trying to parse llm_raw...');
          try {
            let raw = response.data.llm_raw.trim();
            console.log('Raw LLM output (first 200 chars):', raw.substring(0, 200));
            
            // Clean up the raw output
            if (raw.startsWith('```json')) {
              raw = raw.replace(/^```json/, '').replace(/```$/, '').trim();
            } else if (raw.startsWith('```')) {
              raw = raw.replace(/^```/, '').replace(/```$/, '').trim();
            }
            
            console.log('Cleaned raw output (first 200 chars):', raw.substring(0, 200));
            const parsedData = JSON.parse(raw);
            console.log('✅ Parsed data from raw:', parsedData);
            console.log('Available keys in parsed data:', Object.keys(parsedData));
            
            // Fill form with parsed data
            fillFormWithParsedData(parsedData);
          } catch (parseError) {
            console.error('❌ Failed to parse llm_raw:', parseError);
            console.error('Raw content was:', response.data.llm_raw);
            setErrors({ general: 'CV parsing failed - unable to parse AI response. Please try again or fill the form manually.' });
          }
        } else {
          console.error('❌ No valid CV data in response. Expected structured_cv or llm_raw field.');
          console.error('Available fields:', Object.keys(response.data || {}));
          
          // Check if there's a specific error message from the backend
          if (response.data.error) {
            console.error('Backend error:', response.data.error);
            setErrors({ general: `CV parsing failed: ${response.data.error}` });
          } else {
            setErrors({ general: 'CV parsing failed - no structured data returned. Please try again or fill the form manually.' });
          }
        }
      } catch (err) {
        console.error('❌ CV Upload Error:', err);
        if (axios.isAxiosError(err)) {
          console.error('Axios error details:', {
            status: err.response?.status,
            statusText: err.response?.statusText,
            data: err.response?.data,
            headers: err.response?.headers
          });
        }
        
        if (err instanceof Error && err.message.includes('JSON')) {
          setErrors({ general: 'CV parsing failed - invalid data format returned' });
        } else {
          setErrors({ general: 'An error occurred while uploading the CV. Please try again.' });
        }
      } finally {
        setCvLoading(false);
        // Reset the file input to allow re-uploading the same file
        if (fileInputRef.current) {
          fileInputRef.current.value = '';
        }
      }
    }
  };

  // Helper function to consolidate similar skills
  const consolidateSkills = (skills: Skill[]): Skill[] => {
    const skillMap = new Map<string, Skill>();
    
    // Define skill consolidation rules
    const consolidationRules: { [key: string]: string[] } = {
      'JavaScript': ['javascript', 'js', 'ecmascript', 'es6', 'es2015', 'vanilla js'],
      'React': ['react', 'reactjs', 'react.js', 'react hooks', 'redux', 'react router', 'nextjs', 'next.js'],
      'Vue.js': ['vue', 'vuejs', 'vue.js', 'vue 3', 'nuxt', 'nuxtjs'],
      'Angular': ['angular', 'angularjs', 'angular 2+', 'typescript angular'],
      'Node.js': ['nodejs', 'node.js', 'node', 'express', 'expressjs', 'nestjs'],
      'Python': ['python', 'python3', 'py'],
      'Django': ['django', 'django rest framework', 'drf'],
      'Flask': ['flask', 'flask-restful'],
      'FastAPI': ['fastapi', 'fast api'],
      'Java': ['java', 'java 8', 'java 11', 'openjdk'],
      'Spring': ['spring', 'spring boot', 'spring framework', 'spring mvc'],
      'C#': ['c#', 'csharp', 'c sharp', '.net', 'dotnet', 'asp.net'],
      'PHP': ['php', 'php 7', 'php 8', 'laravel', 'symfony', 'codeigniter'],
      'Frontend Development': ['html', 'css', 'html5', 'css3', 'sass', 'scss', 'less', 'bootstrap', 'tailwind', 'tailwindcss', 'material-ui', 'mui'],
      'Database Management': ['sql', 'mysql', 'postgresql', 'postgres', 'sqlite', 'database design', 'database administration'],
      'NoSQL': ['mongodb', 'mongo', 'nosql', 'redis', 'cassandra', 'dynamodb'],
      'Cloud Computing': ['aws', 'amazon web services', 'azure', 'microsoft azure', 'gcp', 'google cloud', 'cloud computing'],
      'DevOps': ['devops', 'ci/cd', 'continuous integration', 'continuous deployment'],
      'Docker': ['docker', 'containerization', 'docker compose'],
      'Kubernetes': ['kubernetes', 'k8s', 'container orchestration'],
      'Git': ['git', 'github', 'gitlab', 'version control', 'bitbucket', 'source control'],
      'Machine Learning': ['ml', 'machine learning', 'ai', 'artificial intelligence', 'deep learning', 'data science'],
      'TensorFlow': ['tensorflow', 'tf', 'keras'],
      'PyTorch': ['pytorch', 'torch'],
      'Data Analysis': ['data analysis', 'pandas', 'numpy', 'matplotlib', 'seaborn', 'data visualization'],
      'Testing': ['testing', 'unit testing', 'jest', 'pytest', 'selenium', 'cypress', 'test automation'],
      'Project Management': ['project management', 'agile', 'scrum', 'kanban', 'jira', 'trello', 'asana'],
      'Communication': ['communication', 'public speaking', 'presentation', 'teamwork', 'collaboration', 'leadership'],
      'Problem Solving': ['problem solving', 'analytical thinking', 'critical thinking', 'debugging'],
      'UI/UX Design': ['ui design', 'ux design', 'user interface', 'user experience', 'figma', 'sketch', 'adobe xd'],
      'Mobile Development': ['mobile development', 'ios', 'android', 'react native', 'flutter', 'swift', 'kotlin'],
      'Cybersecurity': ['cybersecurity', 'security', 'information security', 'penetration testing', 'ethical hacking'],
      'Blockchain': ['blockchain', 'cryptocurrency', 'smart contracts', 'ethereum', 'bitcoin', 'web3']
    };
    
    // First pass: group similar skills
    skills.forEach(skill => {
      const skillName = skill.name.toLowerCase().trim();
      let consolidatedName = skill.name.trim(); // Preserve original case initially
      let consolidatedLevel = skill.level;
      let matched = false;
      
      // Find matching consolidation rule
      for (const [mainSkill, variations] of Object.entries(consolidationRules)) {
        if (variations.includes(skillName) || skillName === mainSkill.toLowerCase()) {
          consolidatedName = mainSkill;
          matched = true;
          break;
        }
        // Check for partial matches (e.g., "react native" contains "react")
        if (variations.some(variation => skillName.includes(variation) || variation.includes(skillName))) {
          consolidatedName = mainSkill;
          matched = true;
          break;
        }
      }
      
      // If no match found, keep original but clean it up
      if (!matched) {
        consolidatedName = skill.name.trim()
          .split(' ')
          .map(word => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
          .join(' ');
      }
      
      // If we already have this consolidated skill, keep the higher level
      if (skillMap.has(consolidatedName)) {
        const existing = skillMap.get(consolidatedName)!;
        const levelPriority = { 'Expert': 4, 'Advanced': 3, 'Intermediate': 2, 'Beginner': 1 };
        if ((levelPriority[skill.level as keyof typeof levelPriority] || 0) > 
            (levelPriority[existing.level as keyof typeof levelPriority] || 0)) {
          consolidatedLevel = skill.level;
        } else {
          consolidatedLevel = existing.level;
        }
      }
      
      skillMap.set(consolidatedName, { name: consolidatedName, level: consolidatedLevel });
    });
    
    // Convert back to array and limit to top skills
    const consolidatedSkills = Array.from(skillMap.values());
    
    // Sort by level priority and limit to 12 skills
    const levelPriority = { 'Expert': 4, 'Advanced': 3, 'Intermediate': 2, 'Beginner': 1 };
    return consolidatedSkills
      .sort((a, b) => (levelPriority[b.level as keyof typeof levelPriority] || 0) - 
                     (levelPriority[a.level as keyof typeof levelPriority] || 0))
      .slice(0, 12);
  };

  // Helper function to fill form with parsed data
  const fillFormWithParsedData = (parsedData: { [key: string]: unknown }) => {
    console.log('Filling form with data:', parsedData);
    
    // Extract different sections based on structure - be flexible with field names
    const basicFields = (parsedData["Basic Profile Fields"] as { [key: string]: unknown }) || parsedData;
    
    // Handle both direct arrays and nested structure for skills
    const skillsSection = parsedData["Skills Array (for normalized skills table)"] as { [key: string]: unknown } | undefined;
    const skillsArrayFromSection = parsedData["Skills Array"] as unknown;
    const skillsArray = 
                       // Try direct array first (new format)
                       (Array.isArray(skillsArrayFromSection) ? skillsArrayFromSection : null) ||
                       // Try nested structure (old format)
                       (skillsSection?.["Skills"] as unknown[]) ||
                       ((skillsArrayFromSection as { [key: string]: unknown })?.["Skills"] as unknown[]) ||
                       (parsedData["Skills"] as unknown[]) || 
                       (parsedData["skills"] as unknown[]) || 
                       (basicFields["Skills"] as unknown[]) || 
                       [];
                       
    // Handle both direct arrays and nested structure for education
    const educationSection = parsedData["Education Array (for normalized education table)"] as { [key: string]: unknown } | undefined;
    const educationArrayFromSection = parsedData["Education Array"] as unknown;
    const educationArray = 
                          // Try direct array first (new format)
                          (Array.isArray(educationArrayFromSection) ? educationArrayFromSection : null) ||
                          // Try nested structure (old format)
                          (educationSection?.["Education"] as unknown[]) ||
                          ((educationArrayFromSection as { [key: string]: unknown })?.["Education"] as unknown[]) ||
                          (parsedData["Education"] as unknown[]) || 
                          (parsedData["education"] as unknown[]) || 
                          (basicFields["Education"] as unknown[]) || 
                          [];
                          
    // Handle both direct arrays and nested structure for experience
    const experienceSection = parsedData["Experience Array (for normalized experience table)"] as { [key: string]: unknown } | undefined;
    const experienceArrayFromSection = parsedData["Experience Array"] as unknown;
    const experienceArray = 
                           // Try direct array first (new format)
                           (Array.isArray(experienceArrayFromSection) ? experienceArrayFromSection : null) ||
                           // Try nested structure (old format)
                           (experienceSection?.["Experience"] as unknown[]) ||
                           ((experienceArrayFromSection as { [key: string]: unknown })?.["Experience"] as unknown[]) ||
                           (parsedData["Experience"] as unknown[]) || 
                           (parsedData["experience"] as unknown[]) || 
                           (basicFields["Experience"] as unknown[]) ||
                           [];
    
    console.log('Extracted arrays:', {
      skills: skillsArray,
      skillsLength: skillsArray.length,
      education: educationArray,
      educationLength: educationArray.length,
      experience: experienceArray,
      experienceLength: experienceArray.length
    });
    
    console.log('First skill:', skillsArray[0]);
    console.log('First education:', educationArray[0]);
    console.log('First experience:', experienceArray[0]);    // Get degree from first education entry
    const firstEducation = Array.isArray(educationArray) && educationArray.length > 0 ? educationArray[0] as { [key: string]: unknown } : null;
    const degree = firstEducation ? (firstEducation.degree as string) || (firstEducation.field as string) || '' : '';
    
    // Format skills for display (legacy field)
    const skillsText = Array.isArray(skillsArray) 
      ? skillsArray.map(skill => typeof skill === 'object' && skill !== null ? (skill as { name?: string }).name || '' : String(skill)).join(', ')
      : '';
    
    // Format languages for display
    const languages = basicFields.Languages as string[] | string | undefined;
    const languagesText = Array.isArray(languages) 
      ? languages.join(', ')
      : languages || '';
    
    // Fill basic form data
    setFormData((prev) => ({
      ...prev,
      fullName: (basicFields.FullName as string) || prev.fullName,
      email: (basicFields.Email as string) || prev.email,
      phone: (basicFields.Phone as string) || prev.phone,
      linkedin: (basicFields.Linkedin as string) || prev.linkedin,
      github: (basicFields.Github as string) || prev.github,
      summary: (basicFields.Summary as string) || prev.summary,
      university: (basicFields.University as string) || prev.university,
      degree: degree || prev.degree,
      graduationYear: (basicFields.GraduationYear as string) || prev.graduationYear,
      skills: skillsText || prev.skills,
      languages: languagesText || prev.languages,
    }));
    
    // Populate experiences array
    if (Array.isArray(experienceArray) && experienceArray.length > 0) {
      const parsedExperiences = experienceArray.map(exp => {
        const experience = exp as { [key: string]: unknown };
        return {
          company: (experience.company as string) || '',
          title: (experience.title as string) || (experience.position as string) || '',
          start_date: (experience.start_date as string) || '',
          end_date: (experience.end_date as string) || '',
          description: (experience.description as string) || ''
        };
      });
      setExperiences(parsedExperiences);
      console.log('Populated experiences:', parsedExperiences);
    }
    
    // Populate skills array
    if (Array.isArray(skillsArray) && skillsArray.length > 0) {
      const parsedSkills = skillsArray.map(skill => {
        if (typeof skill === 'object' && skill !== null) {
          const skillObj = skill as { name?: string; level?: string };
          return {
            name: skillObj.name || '',
            level: skillObj.level || 'Intermediate'
          };
        }
        return {
          name: String(skill),
          level: 'Intermediate'
        };
      });
      
      // Consolidate similar skills
      const consolidatedSkills = consolidateSkills(parsedSkills);
      setSkillsList(consolidatedSkills);
      console.log('Populated and consolidated skills:', consolidatedSkills);
    }

    // Populate education array
    if (Array.isArray(educationArray) && educationArray.length > 0) {
      const parsedEducations = educationArray.map(edu => {
        const education = edu as { [key: string]: unknown };
        return {
          school: (education.school as string) || '',
          degree: (education.degree as string) || '',
          field: (education.field as string) || '',
          start_year: (education.start_year as string) || '',
          end_year: (education.end_year as string) || '',
          description: (education.description as string) || ''
        };
      });
      setEducationsList(parsedEducations);
      console.log('Populated educations:', parsedEducations);
    }
    
    console.log('Form data filled successfully');
  };

  const handleUploadClick = () => {
    fileInputRef.current?.click();
  };

  const handleEmailBlur = async () => {
    if (!formData.email) return;
    setCheckingEmail(true);
    try {
      const res = await apiClient.get(`/api/check-email-exists`, { params: { email: formData.email } });
      if (res.data.exists) {
        setErrors(prev => ({ ...prev, email: 'This email is already used.' }));
      }
    } catch {
      // Optionally handle error
    } finally {
      setCheckingEmail(false);
    }
  };

  if (formData.role === 'recruiter') {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50 flex items-center justify-center py-12 px-4">
        <div className="w-full max-w-2xl bg-white rounded-2xl shadow-xl p-8">
          <div className="mb-8 text-center">
            <div className="mx-auto h-16 w-16 bg-gradient-to-br from-indigo-600 to-purple-600 rounded-full flex items-center justify-center mb-4">
              <FaBuilding className="text-white text-2xl" />
            </div>
            <h2 className="text-2xl font-bold text-gray-900 mb-2">Join PopCareers as Recruiter</h2>
            <p className="text-gray-500">Create your recruiter account to get started</p>
          </div>

          {errors.general && (
            <div className="mb-4 p-4 bg-red-50 border border-red-200 rounded-lg">
              <p className="text-sm text-red-600">{errors.general}</p>
            </div>
          )}

          <form onSubmit={handleSubmit} className="space-y-6">
            {/* Personal Information Section */}
            <div className="space-y-4">
              <h3 className="text-lg font-semibold text-gray-900 border-b pb-2">Personal Information</h3>
              
              <div className="grid md:grid-cols-2 gap-4">
                <div className="relative">
                  <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                    <FaUser />
                  </span>
                  <input
                    type="text"
                    name="fullName"
                    value={formData.fullName}
                    onChange={handleChange}
                    placeholder="Full Name *"
                    className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.fullName ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition`}
                    required
                  />
                  {errors.fullName && <p className="text-xs text-red-600 mt-1">{errors.fullName}</p>}
                </div>

                <div className="relative">
                  <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                    <FaEnvelope />
                  </span>
                  <input
                    type="email"
                    name="email"
                    value={formData.email}
                    onChange={handleChange}
                    onBlur={handleEmailBlur}
                    placeholder="Email Address *"
                    className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.email ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition`}
                    required
                  />
                  {checkingEmail && <span className="text-xs text-gray-500 ml-2">Checking...</span>}
                  {errors.email && <p className="text-xs text-red-600 mt-1">{errors.email}</p>}
                </div>

                <div className="relative">
                  <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                    <FaUser />
                  </span>
                  <input
                    type="text"
                    name="position"
                    value={formData.position}
                    onChange={handleChange}
                    placeholder="Position *"
                    className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.position ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition`}
                    required
                  />
                  {errors.position && <p className="text-xs text-red-600 mt-1">{errors.position}</p>}
                </div>

                <div className="relative">
                  <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                    <FaBuilding />
                  </span>
                  <input
                    type="text"
                    name="department"
                    value={formData.department}
                    onChange={handleChange}
                    placeholder="Department *"
                    className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.department ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition`}
                    required
                  />
                  {errors.department && <p className="text-xs text-red-600 mt-1">{errors.department}</p>}
                </div>
              </div>
            </div>

            {/* Company Information Section */}
            <div className="space-y-4">
              <h3 className="text-lg font-semibold text-gray-900 border-b pb-2">Company Information</h3>
              
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaBuilding />
                </span>
                <input
                  type="text"
                  name="companyName"
                  value={formData.companyName}
                  onChange={handleChange}
                  placeholder="Company Name *"
                  className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.companyName ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition`}
                  required
                />
                {errors.companyName && <p className="text-xs text-red-600 mt-1">{errors.companyName}</p>}
              </div>

              <div className="relative">
                <textarea
                  name="companyDescription"
                  value={formData.companyDescription}
                  onChange={handleChange}
                  placeholder="Company Description (Optional)"
                  rows={3}
                  className="w-full px-4 py-3 rounded-lg border border-gray-300 focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition resize-none"
                />
              </div>

              <div className="grid md:grid-cols-2 gap-4">
                <div className="relative">
                  <input
                    type="text"
                    name="companyIndustry"
                    value={formData.companyIndustry}
                    onChange={handleChange}
                    placeholder="Industry (e.g., Technology)"
                    className="w-full px-4 py-3 rounded-lg border border-gray-300 focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition"
                  />
                </div>

                <div className="relative">
                  <select
                    name="companySize"
                    value={formData.companySize}
                    onChange={handleChange}
                    className="w-full px-4 py-3 rounded-lg border border-gray-300 focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition"
                  >
                    <option value="">Select Company Size</option>
                    <option value="1-10">1-10 employees</option>
                    <option value="11-50">11-50 employees</option>
                    <option value="51-200">51-200 employees</option>
                    <option value="201-500">201-500 employees</option>
                    <option value="501-1000">501-1000 employees</option>
                    <option value="1000+">1000+ employees</option>
                  </select>
                </div>
              </div>

              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaLinkedin />
                </span>
                <input
                  type="url"
                  name="companyWebsite"
                  value={formData.companyWebsite}
                  onChange={handleChange}
                  placeholder="Company Website (e.g., https://www.company.com)"
                  className="w-full pl-10 pr-4 py-3 rounded-lg border border-gray-300 focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition"
                />
              </div>
            </div>

            {/* Password Section */}
            <div className="space-y-4">
              <h3 className="text-lg font-semibold text-gray-900 border-b pb-2">Security</h3>
              
              <div className="grid md:grid-cols-2 gap-4">
                <div className="relative">
                  <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                    <FaLock />
                  </span>
                  <input
                    type="password"
                    name="password"
                    value={formData.password}
                    onChange={handleChange}
                    placeholder="Password *"
                    className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.password ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition`}
                    required
                  />
                  {errors.password && <p className="text-xs text-red-600 mt-1">{errors.password}</p>}
                </div>

                <div className="relative">
                  <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                    <FaLock />
                  </span>
                  <input
                    type="password"
                    name="confirmPassword"
                    value={formData.confirmPassword}
                    onChange={handleChange}
                    placeholder="Confirm Password *"
                    className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.confirmPassword ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none transition`}
                    required
                  />
                  {errors.confirmPassword && <p className="text-xs text-red-600 mt-1">{errors.confirmPassword}</p>}
                </div>
              </div>
            </div>

            <button
              type="submit"
              disabled={loading}
              className="w-full bg-gradient-to-r from-indigo-600 to-purple-600 text-white font-semibold py-3 rounded-lg shadow-lg hover:from-indigo-700 hover:to-purple-700 transition-all duration-200 disabled:opacity-50"
            >
              {loading ? (
                <span className="flex items-center justify-center">
                  <FaSpinner className="animate-spin mr-2" />
                  Creating Account...
                </span>
              ) : (
                'Create Account'
              )}
            </button>
          </form>

          <div className="text-center mt-6">
            <p className="text-sm text-gray-500">
              Already have an account?{' '}
              <Link to="/login" className="text-indigo-600 hover:text-indigo-700 font-medium transition">
                Sign in here
              </Link>
            </p>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50 flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
      <div className="max-w-4xl w-full space-y-8">
        {/* Header */}
        <div className="text-center">
          <div className="mx-auto h-20 w-20 bg-gradient-to-br from-blue-600 to-purple-600 rounded-full flex items-center justify-center mb-6">
            <FaGraduationCap className="text-white text-3xl" />
          </div>
          <h2 className="text-4xl font-bold text-gray-900 mb-2">
            Join PopCareers as Student
          </h2>
          <p className="text-gray-600 text-lg">Create your account to get started</p>
        </div>

        {/* Registration Form */}
        <div className="bg-white rounded-2xl shadow-xl p-8">
          {errors.general && (
            <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg">
              <p className="text-sm text-red-600">{errors.general}</p>
            </div>
          )}

          <form onSubmit={handleSubmit} className="space-y-6">
            {/* CV Upload Section */}
            <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg p-6 border-2 border-dashed border-blue-200">
              <div className="text-center">
                <FaFileUpload className="mx-auto h-12 w-12 text-blue-400 mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">Upload Your CV (Optional)</h3>
                <p className="text-gray-600 mb-4">Upload your CV to automatically fill in your information, experiences, skills, and education</p>
              <input
                type="file"
                accept=".pdf,.doc,.docx"
              onChange={handleCVUpload}
              ref={fileInputRef}
                style={{ display: 'none' }}
              />
                <button
                  type="button"
                  onClick={handleUploadClick}
                  disabled={cvLoading}
                  className="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50"
                >
                  {cvLoading ? (
                    <span className="flex items-center justify-center">
                      <FaSpinner className="animate-spin mr-2" />
                      Processing CV...
                    </span>
                  ) : (
                    'Upload CV'
                  )}
            </button>
              </div>
              
              {/* CV Success Message */}
              {(experiences.length > 0 || skillsList.length > 0 || educationsList.length > 0) && (
                <div className="mt-4 p-4 bg-green-50 border border-green-200 rounded-lg">
                  <div className="flex items-center text-green-800">
                    <FaCheckCircle className="mr-2" />
                    <span className="font-medium">CV data extracted successfully!</span>
                  </div>
                  <div className="mt-2 text-sm text-green-700">
                    Autofilled: {experiences.length} experience(s), {skillsList.length} skill(s), and {educationsList.length} education(s)
                  </div>
                </div>
              )}
            </div>

            {/* Basic Information */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaUser />
                </span>
                <input
                  type="text"
                  name="fullName"
                  value={formData.fullName}
                  onChange={handleChange}
                  placeholder="Full Name"
                  className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.fullName ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition`}
                  required
                />
                {errors.fullName && <p className="text-xs text-red-600 mt-1">{errors.fullName}</p>}
              </div>
              
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaEnvelope />
                </span>
                <input
                  type="email"
                  name="email"
                  value={formData.email}
                  onChange={handleChange}
                  onBlur={handleEmailBlur}
                  placeholder="Email Address"
                  className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.email ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition`}
                  required
                />
                {checkingEmail && <span className="text-xs text-gray-500 ml-2">Checking...</span>}
                {errors.email && <p className="text-xs text-red-600 mt-1">{errors.email}</p>}
              </div>
            </div>

            {/* Contact Information */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaPhone />
                </span>
                <input
                  type="tel"
                  name="phone"
                  value={formData.phone}
                  onChange={handleChange}
                  placeholder="Phone Number"
                  className="w-full pl-10 pr-4 py-3 rounded-lg border border-gray-300 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition"
                />
              </div>
              
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaLinkedin />
                </span>
                <input
                  type="text"
                  name="linkedin"
                  value={formData.linkedin}
                  onChange={handleChange}
                  placeholder="LinkedIn Profile"
                  className="w-full pl-10 pr-4 py-3 rounded-lg border border-gray-300 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition"
                />
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaGithub />
                </span>
                <input
                  type="text"
                  name="github"
                  value={formData.github}
                  onChange={handleChange}
                  placeholder="GitHub Profile"
                  className="w-full pl-10 pr-4 py-3 rounded-lg border border-gray-300 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition"
                />
              </div>
              
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaUniversity />
                </span>
                <input
                  type="text"
                  name="university"
                  value={formData.university}
                  onChange={handleChange}
                  placeholder="University"
                  className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.university ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition`}
                  required
                />
                {errors.university && <p className="text-xs text-red-600 mt-1">{errors.university}</p>}
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaGraduationCap />
                </span>
                <input
                  type="text"
                  name="degree"
                  value={formData.degree}
                  onChange={handleChange}
                  placeholder="Degree/Program"
                  className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.degree ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition`}
                  required
                />
                {errors.degree && <p className="text-xs text-red-600 mt-1">{errors.degree}</p>}
              </div>
              
              <div>
                <input
                  type="number"
                  name="graduationYear"
                  value={formData.graduationYear}
                  onChange={handleChange}
                  placeholder="Graduation Year"
                  min="2000"
                  max="2030"
                  className="w-full px-4 py-3 rounded-lg border border-gray-300 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition"
                />
              </div>
            </div>

            <div>
              <textarea
                name="summary"
                value={formData.summary}
                onChange={handleChange}
                rows={4}
                placeholder="Tell us about yourself, your interests, and career goals..."
                className="w-full px-4 py-3 rounded-lg border border-gray-300 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition resize-none"
              />
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <input
                  type="text"
                  name="languages"
                  value={formData.languages}
                  onChange={handleChange}
                  placeholder="Languages (e.g., English, French, Spanish)"
                  className="w-full px-4 py-3 rounded-lg border border-gray-300 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition"
                />
              </div>
            </div>

            {/* Experiences Section */}
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-semibold text-gray-900">Work Experience</h3>
                <button
                  type="button"
                  onClick={() => openExperienceModal()}
                  className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                >
                  <FaPlus className="mr-2" />
                  Add Experience
                </button>
              </div>
              
              {experiences.length > 0 && (
                <div className="space-y-3">
                  {experiences.map((experience, index) => (
                    <div key={index} className="p-4 border border-gray-200 rounded-lg bg-gray-50">
                      <div className="flex justify-between items-start">
                        <div className="flex-1">
                          <h4 className="font-semibold text-gray-900">{experience.title}</h4>
                          <p className="text-gray-700">{experience.company}</p>
                          <p className="text-sm text-gray-500">
                            {experience.start_date} - {experience.end_date || 'Present'}
                          </p>
                          {experience.description && (
                            <p className="text-sm text-gray-600 mt-2">{experience.description}</p>
                          )}
                        </div>
                        <div className="flex space-x-2">
                          <button
                            type="button"
                            onClick={() => openExperienceModal(experience)}
                            className="text-blue-600 hover:text-blue-800"
                          >
                            <FaEdit />
                          </button>
                          <button
                            type="button"
                            onClick={() => handleDeleteExperience(experience)}
                            className="text-red-600 hover:text-red-800"
                          >
                            <FaTrash />
                          </button>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>

            {/* Skills Section */}
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-semibold text-gray-900">Skills</h3>
                <button
                  type="button"
                  onClick={() => openSkillModal()}
                  className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors"
                >
                  <FaPlus className="mr-2" />
                  Add Skill
                </button>
              </div>
              
              {skillsList.length > 0 && (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                  {skillsList.map((skill, index) => (
                    <div key={index} className="flex items-center justify-between p-3 border border-gray-200 rounded-lg bg-gray-50">
                      <div>
                        <span className="font-medium text-gray-900">{skill.name}</span>
                        <span className="text-sm text-gray-500 ml-2">({skill.level})</span>
                      </div>
                      <div className="flex space-x-2">
                        <button
                          type="button"
                          onClick={() => openSkillModal(skill)}
                          className="text-blue-600 hover:text-blue-800"
                        >
                          <FaEdit />
                        </button>
                        <button
                          type="button"
                          onClick={() => handleDeleteSkill(skill)}
                          className="text-red-600 hover:text-red-800"
                        >
                          <FaTrash />
                        </button>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>

            {/* Education Section */}
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-semibold text-gray-900">Education</h3>
                <button
                  type="button"
                  onClick={() => openEducationModal()}
                  className="flex items-center px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors"
                >
                  <FaPlus className="mr-2" />
                  Add Education
                </button>
              </div>
              
              {educationsList.length > 0 && (
                <div className="space-y-3">
                  {educationsList.map((education, index) => (
                    <div key={index} className="p-4 border border-gray-200 rounded-lg bg-gray-50">
                      <div className="flex items-start justify-between">
                        <div className="flex-1">
                          <h4 className="font-semibold text-gray-900">{education.school}</h4>
                          <p className="text-gray-700">
                            {education.degree} {education.field && `in ${education.field}`}
                          </p>
                          <p className="text-sm text-gray-600">
                            {education.start_year} - {education.end_year || 'Present'}
                          </p>
                          {education.description && (
                            <p className="text-sm text-gray-600 mt-2">{education.description}</p>
                          )}
                        </div>
                        <div className="flex space-x-2 ml-4">
                          <button
                            type="button"
                            onClick={() => openEducationModal(education)}
                            className="text-blue-600 hover:text-blue-800"
                          >
                            <FaEdit />
                          </button>
                          <button
                            type="button"
                            onClick={() => handleDeleteEducation(education)}
                            className="text-red-600 hover:text-red-800"
                          >
                            <FaTrash />
                          </button>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>

            {/* Password fields */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaLock />
                </span>
                <input
                  type="password"
                  name="password"
                  value={formData.password}
                  onChange={handleChange}
                  placeholder="Password"
                  className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.password ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition`}
                  required
                />
                {errors.password && <p className="text-xs text-red-600 mt-1">{errors.password}</p>}
              </div>
              
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400">
                  <FaLock />
                </span>
                <input
                  type="password"
                  name="confirmPassword"
                  value={formData.confirmPassword}
                  onChange={handleChange}
                  placeholder="Confirm Password"
                  className={`w-full pl-10 pr-4 py-3 rounded-lg border ${errors.confirmPassword ? 'border-red-500' : 'border-gray-300'} focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition`}
                  required
                />
                {errors.confirmPassword && <p className="text-xs text-red-600 mt-1">{errors.confirmPassword}</p>}
              </div>
            </div>

          <button 
            type="submit"
              disabled={loading}
              className="w-full bg-gradient-to-r from-blue-600 to-purple-600 text-white font-semibold py-4 rounded-lg shadow-lg hover:from-blue-700 hover:to-purple-700 transition-all duration-200 disabled:opacity-50"
            >
              {loading ? (
                <span className="flex items-center justify-center">
                  <FaSpinner className="animate-spin mr-2" />
                  Creating Account...
                </span>
              ) : (
                'Create Account'
              )}
          </button>
        </form>

          <div className="mt-8 text-center">
            <p className="text-sm text-gray-600">
            Already have an account?{' '}
              <Link to="/login" className="font-semibold text-blue-600 hover:text-blue-500 transition-colors">
                Sign in here
            </Link>
          </p>
          </div>
        </div>
      </div>

      {/* Experience Modal */}
      {showExperienceModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-lg font-semibold">
                {editingExperience ? 'Edit Experience' : 'Add Experience'}
              </h3>
              <button
                type="button"
                onClick={() => setShowExperienceModal(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                <FaTimes />
              </button>
            </div>
            
            <div className="space-y-4">
              <input
                type="text"
                placeholder="Company"
                value={experienceForm.company}
                onChange={(e) => setExperienceForm(prev => ({ ...prev, company: e.target.value }))}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none"
              />
              
              <input
                type="text"
                placeholder="Position"
                value={experienceForm.title}
                onChange={(e) => setExperienceForm(prev => ({ ...prev, title: e.target.value }))}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none"
              />
              
              <div className="grid grid-cols-2 gap-4">
                <input
                  type="date"
                  placeholder="Start Date"
                  value={experienceForm.start_date}
                  onChange={(e) => setExperienceForm(prev => ({ ...prev, start_date: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none"
                />
                
                <input
                  type="date"
                  placeholder="End Date (optional)"
                  value={experienceForm.end_date}
                  onChange={(e) => setExperienceForm(prev => ({ ...prev, end_date: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none"
                />
              </div>
              
              <textarea
                placeholder="Description (optional)"
                value={experienceForm.description}
                onChange={(e) => setExperienceForm(prev => ({ ...prev, description: e.target.value }))}
                rows={3}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none resize-none"
              />
            </div>
            
            <div className="flex justify-end space-x-3 mt-6">
              <button
                type="button"
                onClick={() => setShowExperienceModal(false)}
                className="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50"
              >
                Cancel
              </button>
              <button
                type="button"
                onClick={handleSaveExperience}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
              >
                Save
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Skill Modal */}
      {showSkillModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-lg font-semibold">
                {editingSkill ? 'Edit Skill' : 'Add Skill'}
              </h3>
              <button
                type="button"
                onClick={() => setShowSkillModal(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                <FaTimes />
              </button>
            </div>
            
            <div className="space-y-4">
              <input
                type="text"
                placeholder="Skill Name"
                value={skillForm.name}
                onChange={(e) => setSkillForm(prev => ({ ...prev, name: e.target.value }))}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 outline-none"
              />
              
              <select
                value={skillForm.level}
                onChange={(e) => setSkillForm(prev => ({ ...prev, level: e.target.value }))}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 outline-none"
              >
                <option value="Beginner">Beginner</option>
                <option value="Intermediate">Intermediate</option>
                <option value="Advanced">Advanced</option>
                <option value="Expert">Expert</option>
              </select>
            </div>
            
            <div className="flex justify-end space-x-3 mt-6">
              <button
                type="button"
                onClick={() => setShowSkillModal(false)}
                className="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50"
              >
                Cancel
              </button>
              <button
                type="button"
                onClick={handleSaveSkill}
                className="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700"
              >
                Save
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Education Modal */}
      {showEducationModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold">
                {editingEducation ? 'Edit Education' : 'Add Education'}
              </h3>
              <button
                type="button"
                onClick={() => setShowEducationModal(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                <FaTimes />
              </button>
            </div>
            
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  School/University *
                </label>
                <input
                  type="text"
                  value={educationForm.school}
                  onChange={(e) => setEducationForm(prev => ({ ...prev, school: e.target.value }))}
                  placeholder="Enter school or university name"
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none"
                  required
                />
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Degree
                  </label>
                  <input
                    type="text"
                    value={educationForm.degree || ''}
                    onChange={(e) => setEducationForm(prev => ({ ...prev, degree: e.target.value }))}
                    placeholder="e.g., Bachelor's"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Field of Study
                  </label>
                  <input
                    type="text"
                    value={educationForm.field || ''}
                    onChange={(e) => setEducationForm(prev => ({ ...prev, field: e.target.value }))}
                    placeholder="e.g., Computer Science"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none"
                  />
                </div>
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Start Year
                  </label>
                  <input
                    type="text"
                    value={educationForm.start_year || ''}
                    onChange={(e) => setEducationForm(prev => ({ ...prev, start_year: e.target.value }))}
                    placeholder="e.g., 2020"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    End Year
                  </label>
                  <input
                    type="text"
                    value={educationForm.end_year || ''}
                    onChange={(e) => setEducationForm(prev => ({ ...prev, end_year: e.target.value }))}
                    placeholder="e.g., 2024"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none"
                  />
                </div>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Description (Optional)
                </label>
                <textarea
                  value={educationForm.description || ''}
                  onChange={(e) => setEducationForm(prev => ({ ...prev, description: e.target.value }))}
                  placeholder="Additional details about your education"
                  rows={3}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none resize-none"
                />
              </div>
            </div>
            
            <div className="flex justify-end space-x-3 mt-6">
              <button
                type="button"
                onClick={() => setShowEducationModal(false)}
                className="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50"
              >
                Cancel
              </button>
              <button
                type="button"
                onClick={handleSaveEducation}
                className="px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700"
              >
                Save
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default Register;
