import React, { useState, useRef, useEffect } from 'react';
import { 
  FiCheckCircle,
  FiXCircle,
  FiCamera,
  FiSearch,
  FiZap,
  FiUsers,
  FiEye,
  FiStar,
  FiX,
  FiEdit3,
  FiClock,
  FiTrendingUp
} from 'react-icons/fi';
import apiClient from '../apiClient';
import SmartSearch from '../components/SmartSearch';

interface StudentPreview {
  id: string;
  name: string;
  email: string;
  photo_url?: string;
  university?: string;
  degree?: string;
  skills: Array<{ name: string }>;
}

interface ScanModalProps {
  student: StudentPreview;
  onClose: () => void;
  onSave: (note: string) => void;
  saving: boolean;
}

const ScanModal: React.FC<ScanModalProps> = ({ student, onClose, onSave, saving }) => {
  const [note, setNote] = useState('');

  const topSkills = student.skills.slice(0, 5);

  return (
    <div className="fixed inset-0 bg-black/60 backdrop-blur-sm flex items-center justify-center z-50 p-6 animate-in fade-in duration-300">
      <div className="bg-white/95 backdrop-blur-md rounded-3xl max-w-lg w-full max-h-[90vh] overflow-y-auto border border-white/50 shadow-2xl animate-in slide-in-from-bottom-4 duration-500">
        {/* Enhanced Modal Header */}
        <div className="relative bg-gradient-to-r from-green-600 via-emerald-600 to-teal-700 text-white p-8 rounded-t-3xl overflow-hidden">
          {/* Background Pattern */}
          <div className="absolute inset-0 opacity-10">
            <div 
              className="w-full h-full"
              style={{
                backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23ffffff' fill-opacity='0.05'%3E%3Ccircle cx='30' cy='30' r='4'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`
              }}
            ></div>
          </div>

          <button
            onClick={onClose}
            className="absolute top-6 right-6 w-10 h-10 bg-white/20 backdrop-blur-sm rounded-full flex items-center justify-center hover:bg-white/30 transition-all duration-300 hover:rotate-90"
          >
            <FiX className="text-lg" />
          </button>

          <div className="relative text-center">
            <div className="mb-6">
              <div className="relative inline-block">
                {student.photo_url ? (
                  <img
                    src={student.photo_url}
                    alt={student.name}
                    className="w-24 h-24 rounded-2xl object-cover border-4 border-white/30 shadow-2xl"
                  />
                ) : (
                  <div className="w-24 h-24 rounded-2xl bg-white/20 backdrop-blur-sm flex items-center justify-center text-white text-3xl font-bold border-4 border-white/30 shadow-2xl">
                    {student.name.charAt(0).toUpperCase()}
                  </div>
                )}
                {/* Success indicator */}
                <div className="absolute -bottom-2 -right-2 w-8 h-8 bg-green-400 rounded-full border-3 border-white shadow-lg flex items-center justify-center">
                  <FiCheckCircle className="text-white text-sm" />
                </div>
              </div>
            </div>
            
            <h2 className="text-3xl font-bold mb-2 bg-gradient-to-r from-white to-green-100 bg-clip-text text-transparent">
              {student.name}
            </h2>
            <div className="space-y-1">
              {student.degree && student.university && (
                <p className="text-green-100 text-lg font-medium">
                  {student.degree} • {student.university}
                </p>
              )}
              <p className="text-green-200">{student.email}</p>
            </div>
          </div>
        </div>

        {/* Enhanced Modal Content */}
        <div className="p-8">
          {/* Top Skills */}
          {topSkills.length > 0 && (
            <div className="mb-8">
              <h3 className="text-xl font-bold text-gray-900 mb-4 flex items-center">
                <div className="w-8 h-8 bg-gradient-to-r from-purple-500 to-pink-600 rounded-lg flex items-center justify-center mr-3">
                  <FiTrendingUp className="text-white text-sm" />
                </div>
                Top Skills ({topSkills.length})
              </h3>
              <div className="flex flex-wrap gap-3">
                {topSkills.map((skill, index) => (
                  <span
                    key={index}
                    className="bg-gradient-to-r from-blue-100 to-purple-100 text-blue-800 px-4 py-2 rounded-full text-sm font-medium border border-blue-200/50 backdrop-blur-sm"
                  >
                    {skill.name}
                  </span>
                ))}
              </div>
            </div>
          )}

          {/* Enhanced Note Input */}
          <div className="mb-8">
            <label htmlFor="note" className="text-xl font-bold text-gray-900 mb-4 flex items-center">
              <div className="w-8 h-8 bg-gradient-to-r from-amber-500 to-orange-600 rounded-lg flex items-center justify-center mr-3">
                <FiEdit3 className="text-white text-sm" />
              </div>
              Add Private Note
            </label>
            <div className="bg-amber-50/80 rounded-2xl p-6 border border-amber-100/50 backdrop-blur-sm">
              <textarea
                id="note"
                value={note}
                onChange={(e) => setNote(e.target.value)}
                placeholder="Met at tech fair, interested in full-stack position, great communication skills..."
                className="w-full p-4 border border-amber-200/50 rounded-xl focus:ring-2 focus:ring-amber-500 focus:border-transparent resize-none bg-white/80 backdrop-blur-sm text-gray-800 placeholder-gray-500"
                rows={4}
              />
              <p className="text-amber-700 text-xs mt-2 italic">This note will be private and only visible to you</p>
            </div>
          </div>

          {/* Enhanced Action Buttons */}
          <div className="flex space-x-4">
            <button
              onClick={onClose}
              className="flex-1 bg-gradient-to-r from-gray-500 to-gray-600 text-white py-4 rounded-2xl hover:from-gray-600 hover:to-gray-700 transition-all duration-300 font-semibold flex items-center justify-center space-x-2 shadow-lg hover:shadow-xl transform hover:-translate-y-0.5"
            >
              <FiX className="text-lg" />
              <span>Cancel</span>
            </button>
            <button
              onClick={() => onSave(note)}
              disabled={saving}
              className="flex-1 bg-gradient-to-r from-green-600 to-emerald-600 text-white py-4 rounded-2xl hover:from-green-700 hover:to-emerald-700 transition-all duration-300 font-semibold flex items-center justify-center space-x-2 shadow-lg hover:shadow-xl transform hover:-translate-y-0.5 disabled:opacity-50 disabled:transform-none"
            >
              {saving ? (
                <>
                  <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                  <span>Saving...</span>
                </>
              ) : (
                <>
                  <FiStar className="text-lg" />
                  <span>Save to Shortlist</span>
                </>
              )}
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};

const QRScanner: React.FC = () => {
  const [activeTab, setActiveTab] = useState<'qr' | 'smart'>('qr');
  const [scanning, setScanning] = useState(false);
  const [scanResult, setScanResult] = useState<StudentPreview | null>(null);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [manualUrl, setManualUrl] = useState('');
  const [recentScans, setRecentScans] = useState<StudentPreview[]>([]);
  
  // Toast notification states
  const [notification, setNotification] = useState<{
    show: boolean;
    type: 'success' | 'error';
    message: string;
  }>({ show: false, type: 'success', message: '' });

  // Function to show notification
  const showNotification = (type: 'success' | 'error', message: string) => {
    setNotification({ show: true, type, message });
    setTimeout(() => {
      setNotification({ show: false, type: 'success', message: '' });
    }, 4000);
  };
  
  const videoRef = useRef<HTMLVideoElement | null>(null);
  const canvasRef = useRef<HTMLCanvasElement | null>(null);
  const streamRef = useRef<MediaStream | null>(null);

  useEffect(() => {
    return () => {
      stopCamera();
    };
  }, []);

  const startCamera = async () => {
    setError(null);
    try {
      const stream = await navigator.mediaDevices.getUserMedia({
        video: {
          facingMode: 'environment', // Prefer back camera
          width: { ideal: 1280 },
          height: { ideal: 720 },
        },
      });
      
      streamRef.current = stream;
      if (videoRef.current) {
        videoRef.current.srcObject = stream;
        videoRef.current.play();
        setScanning(true);
        scanQRCode();
      }
    } catch (err) {
      console.error('Error accessing camera:', err);
      setError('Unable to access camera. Please check permissions.');
    }
  };

  const stopCamera = () => {
    if (streamRef.current) {
      streamRef.current.getTracks().forEach(track => track.stop());
      streamRef.current = null;
    }
    setScanning(false);
  };

  const scanQRCode = () => {
    if (!scanning || !videoRef.current || !canvasRef.current) return;

    const video = videoRef.current;
    const canvas = canvasRef.current;
    const context = canvas.getContext('2d');

    if (!context) return;

    // Set canvas size to match video
    canvas.width = video.videoWidth;
    canvas.height = video.videoHeight;

    // Draw current video frame to canvas
    context.drawImage(video, 0, 0, canvas.width, canvas.height);

    // Get image data for QR scanning
    // const imageData = context.getImageData(0, 0, canvas.width, canvas.height);
    
    // Simple URL pattern detection (fallback for manual QR scanning)
    // In a real implementation, you'd use a QR code scanning library like jsQR
    
    // Continue scanning
    if (scanning) {
      requestAnimationFrame(scanQRCode);
    }
  };

  const handleManualUrl = async () => {
    if (!manualUrl.trim()) return;

    // Extract UUID from URL
    const match = manualUrl.match(/\/p\/([a-f0-9-]+)/);
    if (!match) {
      setError('Invalid PopCareers profile URL');
      return;
    }

    const uuid = match[1];
    await fetchStudentProfile(uuid);
  };

  const fetchStudentProfile = async (uuid: string) => {
    setError(null);
    try {
      const response = await apiClient.get(`/api/p/${uuid}`);
      const student = response.data.student;
      
      // Log the QR scan activity
      try {
        await apiClient.post('/api/log-qr-scan', {
          student_id: student.id,
          student_name: student.name
        });
      } catch (logError) {
        console.warn('Failed to log QR scan:', logError);
      }
      
      setScanResult(student);
      setRecentScans(prev => {
        const filtered = prev.filter(s => s.id !== student.id);
        return [student, ...filtered].slice(0, 5); // Keep last 5
      });
      stopCamera();
    } catch (err: unknown) {
      console.error('Error fetching student profile:', err);
      setError('Profile not found or inactive');
    }
  };

  const handleSaveToShortlist = async (note: string) => {
    if (!scanResult) return;

    setSaving(true);
    try {
      await apiClient.post('/api/shortlist', {
        student_id: scanResult.id,
        notes: note || ''
      });
      
      setScanResult(null);
      showNotification('success', 'Candidate successfully added to shortlist!');
    } catch (err: unknown) {
      console.error('Error saving to shortlist:', err);
      const errorResponse = err as { response?: { status: number } };
      if (errorResponse.response?.status === 409) {
        showNotification('error', 'This candidate is already in your shortlist');
      } else {
        showNotification('error', 'Failed to save candidate. Please try again.');
      }
    } finally {
      setSaving(false);
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100">
      {/* Enhanced Toast Notification */}
      {notification.show && (
        <div className={`fixed top-6 right-6 z-50 p-6 rounded-2xl shadow-2xl transition-all duration-500 backdrop-blur-md border transform ${
          notification.type === 'success' 
            ? 'bg-green-500/95 text-white border-green-400/50 animate-in slide-in-from-top-2' 
            : 'bg-red-500/95 text-white border-red-400/50 animate-in slide-in-from-top-2'
        }`}>
          <div className="flex items-center space-x-4">
            <div className="flex-shrink-0">
              <div className={`w-10 h-10 rounded-full flex items-center justify-center ${
                notification.type === 'success' ? 'bg-green-400' : 'bg-red-400'
              }`}>
                {notification.type === 'success' ? (
                  <FiCheckCircle className="text-xl text-white" />
                ) : (
                  <FiXCircle className="text-xl text-white" />
                )}
              </div>
            </div>
            <span className="font-semibold flex-1 text-lg">{notification.message}</span>
            <button
              onClick={() => setNotification({ show: false, type: 'success', message: '' })}
              className="ml-4 text-white hover:text-gray-200 transition-colors w-8 h-8 rounded-full hover:bg-white/20 flex items-center justify-center"
            >
              <FiX className="text-lg" />
            </button>
          </div>
        </div>
      )}
      
      {/* Enhanced Header */}
      <div className="relative overflow-hidden bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-700 text-white">
        <div className="absolute inset-0 bg-black/10"></div>
        <div className="absolute inset-0 opacity-20">
          <div 
            className="w-full h-full"
            style={{
              backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23ffffff' fill-opacity='0.05'%3E%3Ccircle cx='30' cy='30' r='4'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`
            }}
          ></div>
        </div>
        
        <div className="relative max-w-6xl mx-auto px-6 py-16">
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <div className="relative">
                <div className="w-20 h-20 bg-white/20 backdrop-blur-sm rounded-2xl flex items-center justify-center mr-6 border border-white/30">
                  <FiZap className="text-white text-3xl" />
                </div>
                <div className="absolute -inset-1 bg-gradient-to-r from-white/20 to-white/5 rounded-2xl blur-lg"></div>
              </div>
              <div>
                <h1 className="text-5xl font-bold mb-3 bg-gradient-to-r from-white to-blue-100 bg-clip-text text-transparent">
                  {activeTab === 'qr' ? 'QR Code Scanner' : 'Smart Student Search'}
                </h1>
                <p className="text-xl text-blue-100 flex items-center mb-2">
                  <FiUsers className="mr-3 text-lg" />
                  {activeTab === 'qr' ? 'Discover talent instantly at events' : 'AI-powered natural language search'}
                </p>
                <p className="text-blue-200">
                  {activeTab === 'qr' 
                    ? 'Scan student QR codes to view profiles and build your shortlist ⚡'
                    : 'Find candidates using natural language queries powered by AI 🤖'
                  }
                </p>
              </div>
            </div>
            
            {/* Quick Stats */}
            <div className="text-right">
              <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
                <div className="text-3xl font-bold">{recentScans.length}</div>
                <div className="text-blue-100">Recent Scans</div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-6xl mx-auto px-6 py-12">
        {/* Tab Navigation */}
        <div className="flex gap-4 mb-8">
          <button
            onClick={() => setActiveTab('qr')}
            className={`flex-1 py-4 px-6 rounded-2xl font-semibold text-lg transition-all duration-300 ${
              activeTab === 'qr'
                ? 'bg-gradient-to-r from-blue-600 to-indigo-600 text-white shadow-lg'
                : 'bg-white/70 text-gray-700 hover:bg-white/90'
            }`}
          >
            <FiCamera className="inline mr-2" />
            QR Scanner
          </button>
          <button
            onClick={() => setActiveTab('smart')}
            className={`flex-1 py-4 px-6 rounded-2xl font-semibold text-lg transition-all duration-300 ${
              activeTab === 'smart'
                ? 'bg-gradient-to-r from-purple-600 to-pink-600 text-white shadow-lg'
                : 'bg-white/70 text-gray-700 hover:bg-white/90'
            }`}
          >
            <FiZap className="inline mr-2" />
            Smart Search
          </button>
        </div>

        {/* Tab Content */}
        {activeTab === 'smart' ? (
          <SmartSearch />
        ) : (
          <>
        {/* Enhanced Scanner Section */}
        <div className="relative overflow-hidden bg-white/70 backdrop-blur-sm border border-white/50 rounded-3xl p-10 mb-10 shadow-xl">
          <div className="absolute top-0 right-0 w-40 h-40 bg-gradient-to-br from-blue-500/5 to-purple-500/5 rounded-full blur-2xl transform translate-x-20 -translate-y-20"></div>
          
          <div className="relative grid lg:grid-cols-2 gap-12">
            {/* Enhanced Camera Section */}
            <div className="bg-gradient-to-br from-blue-50/80 to-indigo-50/80 rounded-3xl p-8 border border-blue-100/50 backdrop-blur-sm">
              <div className="flex items-center mb-6">
                <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-2xl flex items-center justify-center mr-4">
                  <FiCamera className="text-white text-xl" />
                </div>
                <h2 className="text-2xl font-bold text-blue-900">Camera Scanner</h2>
              </div>
              
              {!scanning ? (
                <div className="text-center">
                  <div className="bg-gradient-to-br from-gray-100 to-gray-200 rounded-2xl h-80 flex items-center justify-center mb-6 border-2 border-dashed border-gray-300 relative overflow-hidden">
                    <div className="absolute inset-0 bg-gradient-to-br from-blue-500/5 to-purple-500/5"></div>
                    <div className="relative text-gray-500">
                      <div className="text-8xl mb-4 animate-pulse">📷</div>
                      <p className="text-lg font-medium">Camera ready to scan</p>
                      <p className="text-sm text-gray-400 mt-2">Point camera at QR code to scan</p>
                    </div>
                  </div>
                  <button
                    onClick={startCamera}
                    className="w-full bg-gradient-to-r from-blue-600 to-indigo-600 text-white py-4 rounded-2xl hover:from-blue-700 hover:to-indigo-700 transition-all duration-300 font-semibold text-lg shadow-lg hover:shadow-xl transform hover:-translate-y-0.5 flex items-center justify-center space-x-3"
                  >
                    <FiCamera className="text-xl" />
                    <span>Start Camera</span>
                  </button>
                </div>
              ) : (
                <div className="text-center">
                  <div className="relative bg-gray-900 rounded-2xl overflow-hidden mb-6 shadow-2xl">
                    <video
                      ref={videoRef}
                      className="w-full h-80 object-cover"
                      autoPlay
                      playsInline
                      muted
                    />
                    <div className="absolute inset-0 flex items-center justify-center">
                      <div className="border-4 border-white/80 w-64 h-64 rounded-2xl shadow-2xl animate-pulse">
                        <div className="absolute top-4 left-4 w-8 h-8 border-t-4 border-l-4 border-white/80 rounded-tl-lg"></div>
                        <div className="absolute top-4 right-4 w-8 h-8 border-t-4 border-r-4 border-white/80 rounded-tr-lg"></div>
                        <div className="absolute bottom-4 left-4 w-8 h-8 border-b-4 border-l-4 border-white/80 rounded-bl-lg"></div>
                        <div className="absolute bottom-4 right-4 w-8 h-8 border-b-4 border-r-4 border-white/80 rounded-br-lg"></div>
                      </div>
                    </div>
                    <div className="absolute bottom-4 left-1/2 transform -translate-x-1/2 bg-black/50 backdrop-blur-sm text-white px-4 py-2 rounded-full text-sm">
                      🔍 Scanning for QR codes...
                    </div>
                  </div>
                  <button
                    onClick={stopCamera}
                    className="w-full bg-gradient-to-r from-red-500 to-red-600 text-white py-4 rounded-2xl hover:from-red-600 hover:to-red-700 transition-all duration-300 font-semibold text-lg shadow-lg hover:shadow-xl transform hover:-translate-y-0.5 flex items-center justify-center space-x-3"
                  >
                    <FiX className="text-xl" />
                    <span>Stop Camera</span>
                  </button>
                </div>
              )}

              <canvas ref={canvasRef} style={{ display: 'none' }} />
            </div>

            {/* Enhanced Manual Input Section */}
            <div className="bg-gradient-to-br from-green-50/80 to-emerald-50/80 rounded-3xl p-8 border border-green-100/50 backdrop-blur-sm">
              <div className="flex items-center mb-6">
                <div className="w-12 h-12 bg-gradient-to-br from-green-500 to-emerald-600 rounded-2xl flex items-center justify-center mr-4">
                  <FiSearch className="text-white text-xl" />
                </div>
                <h2 className="text-2xl font-bold text-green-900">Manual Entry</h2>
              </div>
              
              <div className="space-y-6">
                <div>
                  <label htmlFor="manual-url" className="block text-lg font-semibold text-green-900 mb-3">
                    Paste Profile URL
                  </label>
                  <div className="relative">
                    <input
                      id="manual-url"
                      type="url"
                      value={manualUrl}
                      onChange={(e) => setManualUrl(e.target.value)}
                      placeholder="https://popcareer.com/p/student-uuid..."
                      className="w-full p-4 pr-12 border border-green-200/50 rounded-2xl focus:ring-2 focus:ring-green-500 focus:border-transparent bg-white/80 backdrop-blur-sm text-gray-800 placeholder-gray-500"
                    />
                    <FiSearch className="absolute right-4 top-1/2 transform -translate-y-1/2 text-green-500 text-xl" />
                  </div>
                </div>
                
                <button
                  onClick={handleManualUrl}
                  disabled={!manualUrl.trim()}
                  className="w-full bg-gradient-to-r from-green-600 to-emerald-600 text-white py-4 rounded-2xl hover:from-green-700 hover:to-emerald-700 transition-all duration-300 font-semibold text-lg shadow-lg hover:shadow-xl transform hover:-translate-y-0.5 disabled:opacity-50 disabled:transform-none flex items-center justify-center space-x-3"
                >
                  <FiEye className="text-xl" />
                  <span>Load Profile</span>
                </button>

                {error && (
                  <div className="p-4 bg-red-50/80 border border-red-200/50 rounded-2xl backdrop-blur-sm animate-in slide-in-from-top-2">
                    <div className="flex items-center space-x-3">
                      <FiXCircle className="text-red-500 text-xl flex-shrink-0" />
                      <p className="text-red-700 font-medium">{error}</p>
                    </div>
                  </div>
                )}
                
                {/* Help Text */}
                <div className="bg-green-100/80 rounded-2xl p-4 border border-green-200/50">
                  <h4 className="font-semibold text-green-900 mb-2">💡 How to get profile URLs:</h4>
                  <ul className="text-green-800 text-sm space-y-1">
                    <li>• Ask students to share their PopCareers profile link</li>
                    <li>• Copy URL from student's QR code display</li>
                    <li>• Use when QR scanning isn't available</li>
                  </ul>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Enhanced Recent Scans */}
        {recentScans.length > 0 && (
          <div className="relative overflow-hidden bg-white/70 backdrop-blur-sm border border-white/50 rounded-3xl p-10 shadow-xl">
            <div className="absolute top-0 left-0 w-40 h-40 bg-gradient-to-br from-purple-500/5 to-pink-500/5 rounded-full blur-2xl transform -translate-x-20 -translate-y-20"></div>
            
            <div className="relative">
              <h2 className="text-3xl font-bold text-gray-900 mb-8 flex items-center">
                <div className="w-12 h-12 bg-gradient-to-br from-purple-500 to-pink-600 rounded-2xl flex items-center justify-center mr-4">
                  <FiClock className="text-white text-xl" />
                </div>
                Recent Scans ({recentScans.length})
              </h2>
              
              <div className="grid gap-6">
                {recentScans.map((student, index) => (
                  <div
                    key={student.id}
                    className="group bg-gradient-to-r from-white/80 to-gray-50/80 backdrop-blur-sm rounded-2xl p-6 border border-gray-200/50 hover:border-purple-200/50 hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1"
                    style={{
                      animationDelay: `${index * 100}ms`,
                      animation: 'fadeInUp 0.6s ease-out forwards'
                    }}
                  >
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-4">
                        <div className="relative">
                          {student.photo_url ? (
                            <img
                              src={student.photo_url}
                              alt={student.name}
                              className="w-16 h-16 rounded-2xl object-cover border-2 border-white shadow-lg group-hover:shadow-xl transition-shadow duration-300"
                            />
                          ) : (
                            <div className="w-16 h-16 rounded-2xl bg-gradient-to-br from-purple-500 to-pink-600 flex items-center justify-center text-white text-xl font-bold shadow-lg group-hover:shadow-xl transition-shadow duration-300">
                              {student.name.charAt(0).toUpperCase()}
                            </div>
                          )}
                          {/* Recent scan indicator */}
                          <div className="absolute -top-1 -right-1 w-6 h-6 bg-green-500 rounded-full border-2 border-white shadow-md flex items-center justify-center">
                            <div className="w-2 h-2 bg-white rounded-full animate-pulse"></div>
                          </div>
                        </div>
                        
                        <div className="flex-1">
                          <h3 className="font-bold text-gray-900 text-lg group-hover:text-purple-600 transition-colors duration-300">
                            {student.name}
                          </h3>
                          <p className="text-gray-600 mb-1">{student.university}</p>
                          <p className="text-gray-500 text-sm">{student.email}</p>
                        </div>
                        
                        {/* Skills preview */}
                        {student.skills.length > 0 && (
                          <div className="hidden sm:flex flex-wrap gap-2 max-w-xs">
                            {student.skills.slice(0, 3).map((skill, skillIndex) => (
                              <span
                                key={skillIndex}
                                className="bg-purple-100 text-purple-800 px-3 py-1 rounded-full text-xs font-medium border border-purple-200"
                              >
                                {skill.name}
                              </span>
                            ))}
                            {student.skills.length > 3 && (
                              <span className="bg-gray-100 text-gray-600 px-3 py-1 rounded-full text-xs">
                                +{student.skills.length - 3}
                              </span>
                            )}
                          </div>
                        )}
                      </div>
                      
                      <button
                        onClick={() => setScanResult(student)}
                        className="bg-gradient-to-r from-purple-600 to-pink-600 text-white px-6 py-3 rounded-2xl hover:from-purple-700 hover:to-pink-700 transition-all duration-300 font-semibold flex items-center space-x-2 shadow-lg hover:shadow-xl transform hover:-translate-y-0.5"
                      >
                        <FiEye className="text-lg" />
                        <span>View Again</span>
                      </button>
                    </div>
                  </div>
                ))}
              </div>
              
              <div className="mt-8 text-center">
                <p className="text-gray-600 text-sm">
                  🕒 Showing your {recentScans.length} most recent scans • Scan more profiles to build your talent pipeline
                </p>
              </div>
            </div>
          </div>
        )}

        {/* Scan Result Modal */}
        {scanResult && (
          <ScanModal
            student={scanResult}
            onClose={() => setScanResult(null)}
            onSave={handleSaveToShortlist}
            saving={saving}
          />
        )}
          </>
        )}
      </div>
    </div>
  );
};

export default QRScanner;
