import React, { useState, useEffect } from 'react';
import { QRCodeSVG } from 'qrcode.react';
import { 
  FiDownload, 
  FiShare2, 
  FiCheckCircle, 
  FiXCircle,
  FiTarget,
  FiSettings,
  FiMail,
  FiMapPin,
  FiBook,
  FiExternalLink,
  FiRefreshCw,
  FiAlertCircle
} from 'react-icons/fi';
import apiClient from '../apiClient';

interface Student {
  id: string;
  name: string;
  email: string;
  photo_url?: string;
  university?: string;
  degree?: string;
  qr_active: boolean;
}

const MyCard: React.FC = () => {
  const [student, setStudent] = useState<Student | null>(null);
  const [loading, setLoading] = useState(true);
  const [updating, setUpdating] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const publicProfileUrl = student ? `${window.location.origin}/p/${student.id}` : '';

  useEffect(() => {
    fetchProfile();
  }, []);

  const fetchProfile = async () => {
    try {
      const response = await apiClient.get('/api/user');
      const userData = response.data.data?.user || response.data.user || response.data;
      
      // Map the response to student data structure
      const studentData: Student = {
        id: userData.id,
        name: userData.profile?.full_name || userData.full_name || userData.name || 'Unknown User',
        email: userData.email,
        photo_url: userData.profile?.photo_url || userData.photo_url,
        university: userData.profile?.university || userData.university,
        degree: userData.profile?.degree || userData.degree,
        qr_active: userData.profile?.is_card_active || userData.is_card_active || false,
      };
      
      setStudent(studentData);
    } catch (err) {
      console.error('Error fetching profile:', err);
      setError('Failed to load profile');
    } finally {
      setLoading(false);
    }
  };

  const toggleQRStatus = async () => {
    if (!student) return;

    setUpdating(true);
    try {
      await apiClient.patch('/api/card-activation', {
        is_card_active: !student.qr_active
      });
      
      // Update the local state
      setStudent({
        ...student,
        qr_active: !student.qr_active
      });
    } catch (err) {
      console.error('Error updating QR status:', err);
      alert('Failed to update QR code status');
    } finally {
      setUpdating(false);
    }
  };

  const downloadQR = () => {
    const svg = document.getElementById('qr-code');
    if (!svg) return;

    const svgData = new XMLSerializer().serializeToString(svg);
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    const img = new Image();
    
    img.onload = () => {
      canvas.width = img.width;
      canvas.height = img.height;
      ctx?.drawImage(img, 0, 0);
      
      const pngFile = canvas.toDataURL('image/png');
      const downloadLink = document.createElement('a');
      downloadLink.download = `popcareer-qr-${student?.name?.replace(/\s+/g, '-').toLowerCase() || 'profile'}.png`;
      downloadLink.href = pngFile;
      downloadLink.click();
    };
    
    img.src = 'data:image/svg+xml;base64,' + btoa(svgData);
  };

  const shareProfile = async () => {
    if (!publicProfileUrl) return;

    if (navigator.share) {
      try {
        await navigator.share({
          title: `${student?.name}'s Professional Profile`,
          text: `Check out my professional profile on PopCareers`,
          url: publicProfileUrl,
        });
      } catch (err) {
        console.log('Error sharing:', err);
      }
    } else {
      // Fallback: copy to clipboard
      try {
        await navigator.clipboard.writeText(publicProfileUrl);
        alert('Profile link copied to clipboard!');
      } catch (err) {
        console.log('Error copying to clipboard:', err);
        alert('Unable to copy link');
      }
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="w-12 h-12 border-4 border-blue-200 border-t-blue-600 rounded-full animate-spin mx-auto"></div>
          <p className="mt-4 text-gray-600 font-medium">Loading your card...</p>
        </div>
      </div>
    );
  }

  if (error || !student) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <FiAlertCircle className="text-red-600 text-2xl" />
          </div>
          <h1 className="text-2xl font-bold text-gray-900 mb-2">Error Loading Card</h1>
          <p className="text-gray-600 mb-4">{error || 'Failed to load profile'}</p>
          <button 
            onClick={fetchProfile}
            className="flex items-center space-x-2 bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors mx-auto"
          >
            <FiRefreshCw className="text-sm" />
            <span>Try Again</span>
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 py-12">
      <div className="max-w-7xl mx-auto px-4">
        {/* Ultra Modern Header */}
        <div className="text-center mb-12">
          <div className="flex items-center justify-center mb-6">
            <div className="relative">
              <div className="w-16 h-16 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl flex items-center justify-center shadow-xl">
                <FiTarget className="text-white text-2xl" />
              </div>
              <div className="absolute -inset-1 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl blur-lg opacity-30 animate-pulse"></div>
            </div>
          </div>
          <h1 className="text-5xl font-bold mb-4">
            <span className="bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 bg-clip-text text-transparent">
              My Digital Card
            </span>
          </h1>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
            Your premium digital business card powered by cutting-edge QR technology.
            <br />
            <span className="text-gray-500">Share your professional identity in seconds.</span>
          </p>
        </div>

        {/* Enhanced Status Banner */}
        <div className="max-w-4xl mx-auto mb-12">
          <div className={`relative overflow-hidden rounded-2xl p-6 backdrop-blur-lg border ${
            student.qr_active 
              ? 'bg-gradient-to-r from-emerald-50/80 to-green-50/80 border-emerald-200/50' 
              : 'bg-gradient-to-r from-amber-50/80 to-yellow-50/80 border-amber-200/50'
          }`}>
            <div className="relative z-10 flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <div className={`relative w-14 h-14 rounded-xl flex items-center justify-center ${
                  student.qr_active ? 'bg-emerald-500' : 'bg-amber-500'
                } shadow-lg`}>
                  {student.qr_active ? (
                    <FiCheckCircle className="text-white text-xl" />
                  ) : (
                    <FiXCircle className="text-white text-xl" />
                  )}
                  <div className={`absolute inset-0 rounded-xl ${
                    student.qr_active ? 'bg-emerald-500' : 'bg-amber-500'
                  } animate-ping opacity-20`}></div>
                </div>
                <div>
                  <h3 className="text-xl font-bold text-gray-900 mb-1">
                    Card Status: {student.qr_active ? 'Live & Active' : 'Inactive'}
                  </h3>
                  <p className="text-gray-600">
                    {student.qr_active 
                      ? '🟢 Ready for scanning at networking events' 
                      : '🟡 Card is dormant - activate to enable scanning'
                    }
                  </p>
                </div>
              </div>
              <button
                onClick={toggleQRStatus}
                disabled={updating}
                className={`group relative px-6 py-3 rounded-xl font-semibold transition-all duration-300 transform hover:scale-105 ${
                  student.qr_active
                    ? 'bg-gradient-to-r from-amber-500 to-orange-500 hover:from-amber-600 hover:to-orange-600 text-white shadow-lg'
                    : 'bg-gradient-to-r from-emerald-500 to-green-500 hover:from-emerald-600 hover:to-green-600 text-white shadow-lg'
                } disabled:opacity-50 disabled:transform-none`}
              >
                <div className="flex items-center space-x-2">
                  <FiSettings className={`text-lg transition-transform duration-300 ${updating ? 'animate-spin' : 'group-hover:rotate-90'}`} />
                  <span>{updating ? 'Updating...' : student.qr_active ? 'Deactivate' : 'Activate'}</span>
                </div>
                <div className="absolute inset-0 rounded-xl bg-white/20 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
              </button>
            </div>
          </div>
        </div>

        {/* Premium Business Card Display */}
        <div className="max-w-5xl mx-auto mb-16">
          <div className="relative">
            {/* Background Decorations */}
            <div className="absolute -top-6 -left-6 w-72 h-72 bg-gradient-to-br from-blue-400/20 to-purple-400/20 rounded-full blur-3xl"></div>
            <div className="absolute -bottom-6 -right-6 w-72 h-72 bg-gradient-to-br from-indigo-400/20 to-pink-400/20 rounded-full blur-3xl"></div>
            
            {/* Main Card Container */}
            <div className="relative bg-white/70 backdrop-blur-xl border border-white/30 rounded-3xl shadow-2xl overflow-hidden">
              <div className="p-8 lg:p-12">
                <div className="grid lg:grid-cols-3 gap-8 items-center">
                  {/* Profile Section */}
                  <div className="lg:col-span-2 space-y-6">
                    <div className="flex items-start space-x-6">
                      <div className="relative">
                        <div className="w-24 h-24 lg:w-32 lg:h-32 rounded-2xl overflow-hidden ring-4 ring-white/50 shadow-xl">
                          {student.photo_url ? (
                            <img
                              src={student.photo_url}
                              alt={student.name || 'User'}
                              className="w-full h-full object-cover"
                            />
                          ) : (
                            <div className="w-full h-full bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center text-white text-2xl lg:text-3xl font-bold">
                              {student.name?.charAt(0)?.toUpperCase() || 'U'}
                            </div>
                          )}
                        </div>
                        <div className="absolute -bottom-2 -right-2 w-8 h-8 bg-emerald-500 rounded-full flex items-center justify-center ring-4 ring-white">
                          <FiCheckCircle className="text-white text-sm" />
                        </div>
                      </div>
                      
                      <div className="flex-1 space-y-4">
                        <div>
                          <h2 className="text-3xl lg:text-4xl font-bold text-gray-900 mb-2">
                            {student.name || 'Unknown User'}
                          </h2>
                          <p className="text-xl text-gray-600 font-medium">
                            {student.degree || 'Student'}
                          </p>
                        </div>
                        
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                          {student.university && (
                            <div className="flex items-center space-x-3 p-3 rounded-xl bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-100">
                              <div className="w-10 h-10 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-lg flex items-center justify-center">
                                <FiBook className="text-white text-sm" />
                              </div>
                              <div>
                                <p className="text-sm font-semibold text-gray-900">{student.university}</p>
                                <p className="text-xs text-gray-600">Education</p>
                              </div>
                            </div>
                          )}
                          
                          <div className="flex items-center space-x-3 p-3 rounded-xl bg-gradient-to-r from-purple-50 to-pink-50 border border-purple-100">
                            <div className="w-10 h-10 bg-gradient-to-r from-purple-500 to-pink-500 rounded-lg flex items-center justify-center">
                              <FiMapPin className="text-white text-sm" />
                            </div>
                            <div>
                              <p className="text-sm font-semibold text-gray-900">Available Worldwide</p>
                              <p className="text-xs text-gray-600">Location</p>
                            </div>
                          </div>
                          
                          <div className="sm:col-span-2 flex items-center space-x-3 p-3 rounded-xl bg-gradient-to-r from-emerald-50 to-green-50 border border-emerald-100">
                            <div className="w-10 h-10 bg-gradient-to-r from-emerald-500 to-green-500 rounded-lg flex items-center justify-center">
                              <FiMail className="text-white text-sm" />
                            </div>
                            <div>
                              <p className="text-sm font-semibold text-gray-900">{student.email}</p>
                              <p className="text-xs text-gray-600">Contact</p>
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>
                  
                  {/* QR Code Section */}
                  <div className="flex flex-col items-center space-y-6">
                    <div className="relative group">
                      <div className="absolute -inset-4 bg-gradient-to-r from-blue-500 via-purple-500 to-indigo-500 rounded-2xl blur-lg opacity-30 group-hover:opacity-50 transition-opacity duration-300"></div>
                      <div className="relative bg-white p-6 rounded-2xl shadow-xl ring-1 ring-gray-100 transform group-hover:scale-105 transition-transform duration-300">
                        <QRCodeSVG
                          id="qr-code"
                          value={publicProfileUrl}
                          size={160}
                          bgColor="#ffffff"
                          fgColor="#1e40af"
                          level="H"
                          includeMargin={false}
                        />
                      </div>
                    </div>
                    
                    <div className="text-center space-y-2">
                      <h3 className="text-lg font-bold text-gray-900">Scan to Connect</h3>
                      <p className="text-sm text-gray-600 max-w-32">Quick access to my professional profile</p>
                    </div>
                  </div>
                </div>
              </div>
              
              {/* Card Footer */}
              <div className="bg-gradient-to-r from-blue-500/5 via-purple-500/5 to-indigo-500/5 px-8 py-6 border-t border-gray-100">
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-2">
                    <FiTarget className="text-blue-600 text-lg" />
                    <span className="text-sm font-semibold text-gray-900">PopCareers</span>
                    <span className="text-xs text-gray-500">Digital Business Card</span>
                  </div>
                  <div className="text-xs text-gray-400">
                    ID: {student.id}
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Enhanced Action Buttons */}
        <div className="max-w-4xl mx-auto mb-16">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <button
              onClick={downloadQR}
              className="group relative overflow-hidden bg-white backdrop-blur-lg border border-gray-200/50 rounded-2xl p-6 shadow-lg hover:shadow-2xl transition-all duration-300 transform hover:scale-105"
            >
              <div className="absolute inset-0 bg-gradient-to-r from-blue-500/5 to-purple-500/5 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
              <div className="relative flex flex-col items-center space-y-4">
                <div className="w-14 h-14 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-xl flex items-center justify-center shadow-lg group-hover:shadow-xl transition-shadow duration-300">
                  <FiDownload className="text-white text-xl" />
                </div>
                <div className="text-center">
                  <h3 className="text-lg font-bold text-gray-900 mb-1">Download QR</h3>
                  <p className="text-sm text-gray-600">Save QR code as PNG image</p>
                </div>
              </div>
            </button>

            <button
              onClick={shareProfile}
              className="group relative overflow-hidden bg-white backdrop-blur-lg border border-gray-200/50 rounded-2xl p-6 shadow-lg hover:shadow-2xl transition-all duration-300 transform hover:scale-105"
            >
              <div className="absolute inset-0 bg-gradient-to-r from-purple-500/5 to-pink-500/5 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
              <div className="relative flex flex-col items-center space-y-4">
                <div className="w-14 h-14 bg-gradient-to-r from-purple-500 to-pink-500 rounded-xl flex items-center justify-center shadow-lg group-hover:shadow-xl transition-shadow duration-300">
                  <FiShare2 className="text-white text-xl" />
                </div>
                <div className="text-center">
                  <h3 className="text-lg font-bold text-gray-900 mb-1">Share Profile</h3>
                  <p className="text-sm text-gray-600">Send your profile link directly</p>
                </div>
              </div>
            </button>

            <a
              href={publicProfileUrl}
              target="_blank"
              rel="noopener noreferrer"
              className="group relative overflow-hidden bg-white backdrop-blur-lg border border-gray-200/50 rounded-2xl p-6 shadow-lg hover:shadow-2xl transition-all duration-300 transform hover:scale-105"
            >
              <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/5 to-green-500/5 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
              <div className="relative flex flex-col items-center space-y-4">
                <div className="w-14 h-14 bg-gradient-to-r from-emerald-500 to-green-500 rounded-xl flex items-center justify-center shadow-lg group-hover:shadow-xl transition-shadow duration-300">
                  <FiExternalLink className="text-white text-xl" />
                </div>
                <div className="text-center">
                  <h3 className="text-lg font-bold text-gray-900 mb-1">View Live</h3>
                  <p className="text-sm text-gray-600">See your public profile page</p>
                </div>
              </div>
            </a>
          </div>
        </div>

        {/* Information Panels */}
        <div className="max-w-4xl mx-auto mb-16">
          <div className="grid md:grid-cols-2 gap-8">
            {/* How It Works Panel */}
            <div className="relative overflow-hidden bg-white/60 backdrop-blur-lg border border-gray-200/50 rounded-2xl p-8 shadow-lg">
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-blue-500 to-purple-500"></div>
              <div className="flex items-center mb-6">
                <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-xl flex items-center justify-center mr-4">
                  <FiTarget className="text-white text-xl" />
                </div>
                <h3 className="text-xl font-bold text-gray-900">How It Works</h3>
              </div>
              <div className="space-y-4">
                <div className="flex items-start space-x-3">
                  <div className="w-6 h-6 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0 mt-0.5">
                    <span className="text-blue-600 text-xs font-bold">1</span>
                  </div>
                  <p className="text-gray-700 text-sm">Recruiters scan your QR code at networking events</p>
                </div>
                <div className="flex items-start space-x-3">
                  <div className="w-6 h-6 bg-purple-100 rounded-full flex items-center justify-center flex-shrink-0 mt-0.5">
                    <span className="text-purple-600 text-xs font-bold">2</span>
                  </div>
                  <p className="text-gray-700 text-sm">They instantly access your complete professional profile</p>
                </div>
                <div className="flex items-start space-x-3">
                  <div className="w-6 h-6 bg-emerald-100 rounded-full flex items-center justify-center flex-shrink-0 mt-0.5">
                    <span className="text-emerald-600 text-xs font-bold">3</span>
                  </div>
                  <p className="text-gray-700 text-sm">You get notified when someone views your profile</p>
                </div>
              </div>
            </div>

            {/* Settings Panel */}
            <div className="relative overflow-hidden bg-white/60 backdrop-blur-lg border border-gray-200/50 rounded-2xl p-8 shadow-lg">
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-emerald-500 to-green-500"></div>
              <div className="flex items-center mb-6">
                <div className="w-12 h-12 bg-gradient-to-r from-emerald-500 to-green-500 rounded-xl flex items-center justify-center mr-4">
                  <FiSettings className="text-white text-xl" />
                </div>
                <h3 className="text-xl font-bold text-gray-900">Card Settings</h3>
              </div>
              <div className="space-y-4">
                <div className="flex items-start space-x-3">
                  <div className="w-6 h-6 bg-emerald-100 rounded-full flex items-center justify-center flex-shrink-0 mt-0.5">
                    <FiCheckCircle className="text-emerald-600 text-xs" />
                  </div>
                  <p className="text-gray-700 text-sm">Toggle card active/inactive status anytime</p>
                </div>
                <div className="flex items-start space-x-3">
                  <div className="w-6 h-6 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0 mt-0.5">
                    <FiDownload className="text-blue-600 text-xs" />
                  </div>
                  <p className="text-gray-700 text-sm">Download QR code for printing on business cards</p>
                </div>
                <div className="flex items-start space-x-3">
                  <div className="w-6 h-6 bg-purple-100 rounded-full flex items-center justify-center flex-shrink-0 mt-0.5">
                    <FiShare2 className="text-purple-600 text-xs" />
                  </div>
                  <p className="text-gray-700 text-sm">Share your profile link via email or social media</p>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Profile URL Display */}
        <div className="max-w-3xl mx-auto">
          <div className="bg-gradient-to-r from-gray-50 to-blue-50 border border-gray-200 rounded-2xl p-6">
            <div className="flex items-center space-x-3 mb-4">
              <div className="w-10 h-10 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-lg flex items-center justify-center">
                <FiExternalLink className="text-white text-sm" />
              </div>
              <h3 className="text-lg font-bold text-gray-900">Your Profile URL</h3>
            </div>
            <div className="bg-white rounded-xl p-4 border border-gray-200">
              <p className="text-sm text-gray-600 font-mono break-all bg-gray-50 p-3 rounded-lg">
                {publicProfileUrl}
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default MyCard;
