import React, { useState } from 'react';
import { FiSearch, FiX, FiUser, FiAward, FiBookOpen, FiBriefcase } from 'react-icons/fi';
import { useNavigate } from 'react-router-dom';
import apiClient from '../apiClient';

interface SearchCriteria {
  skills: string[];
  university: string | null;
  degree: string | null;
  field_of_study: string | null;
  experience_level: string | null;
  min_experience_years: number | null;
}

interface MatchedProfile {
  id: string;
  score: number;
  reason: string;
  profile: {
    full_name: string;
    email: string;
    phone: string | null;
    bio: string | null;
    profile_image: string | null;
    skills: Array<{ id: string; name: string; proficiency_level: string | null }>;
    educations: Array<{
      university: string;
      degree_type: string;
      field_of_study: string;
      start_date: string;
      end_date: string | null;
    }>;
    experiences: Array<{
      job_title: string;
      company_name: string;
      description: string | null;
      start_date: string;
      end_date: string | null;
    }>;
  };
  matching_criteria: string[];
}

interface SearchResult {
  query: string;
  extracted_criteria: SearchCriteria;
  results: MatchedProfile[];
  total: number;
  page: number;
  per_page: number;
  total_pages: number;
}

const SmartSearch: React.FC = () => {
  const [query, setQuery] = useState('');
  const [loading, setLoading] = useState(false);
  const [searchResult, setSearchResult] = useState<SearchResult | null>(null);
  const [error, setError] = useState<string | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [shortlistingId, setShortlistingId] = useState<string | null>(null);
  const [successMessage, setSuccessMessage] = useState<string | null>(null);
  const navigate = useNavigate();

  const exampleQueries = [
    'Python developer from ENSAM with machine learning experience',
    'Frontend engineer skilled in React and TypeScript',
    'Data scientist with PhD in computer science',
    'Full-stack developer with 3+ years of experience',
    'Junior developer from MIT with Java skills',
  ];

  const handleSearch = async (page: number = 1) => {
    if (!query.trim()) {
      setError('Please enter a search query');
      return;
    }

    setLoading(true);
    setError(null);

    try {
      const response = await apiClient.post('/api/students/smart-search', {
        query: query.trim(),
        page,
        per_page: 10,
      });

      if (response.data.success) {
        setSearchResult(response.data.data);
        setCurrentPage(page);
      } else {
        setError('Search failed. Please try again.');
      }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    } catch (err: any) {
      console.error('Smart search error:', err);
      setError(err.response?.data?.message || 'An error occurred during search');
    } finally {
      setLoading(false);
    }
  };

  const handleExampleClick = (exampleQuery: string) => {
    setQuery(exampleQuery);
    setSearchResult(null);
    setError(null);
  };

  const handleClearSearch = () => {
    setQuery('');
    setSearchResult(null);
    setError(null);
    setCurrentPage(1);
  };

  const handleViewProfile = (studentId: string) => {
    navigate(`/scan-profile/${studentId}`);
  };

  const handleAddToShortlist = async (studentId: string, studentName: string) => {
    setShortlistingId(studentId);
    setError(null);
    setSuccessMessage(null);

    try {
      await apiClient.post('/api/shortlist', {
        student_id: studentId,
        private_note: `Added from Smart Search: ${query}`,
      });

      setSuccessMessage(`${studentName} added to shortlist!`);
      setTimeout(() => setSuccessMessage(null), 3000);
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    } catch (err: any) {
      console.error('Add to shortlist error:', err);
      if (err.response?.status === 409) {
        setError('This candidate is already in your shortlist');
      } else {
        setError(err.response?.data?.message || 'Failed to add to shortlist');
      }
      setTimeout(() => setError(null), 3000);
    } finally {
      setShortlistingId(null);
    }
  };

  const getScoreColor = (score: number): string => {
    if (score >= 80) return 'text-green-600 bg-green-50 border-green-200';
    if (score >= 60) return 'text-blue-600 bg-blue-50 border-blue-200';
    if (score >= 40) return 'text-yellow-600 bg-yellow-50 border-yellow-200';
    return 'text-gray-600 bg-gray-50 border-gray-200';
  };

  const getScoreLabel = (score: number): string => {
    if (score >= 80) return 'Excellent Match';
    if (score >= 60) return 'Good Match';
    if (score >= 40) return 'Partial Match';
    return 'Low Match';
  };

  return (
    <div className="w-full max-w-6xl mx-auto p-6">
      {/* Header */}
      <div className="mb-8">
        <h2 className="text-3xl font-bold bg-gradient-to-r from-purple-600 to-pink-600 bg-clip-text text-transparent mb-2">
          Smart Student Search
        </h2>
        <p className="text-gray-600">
          Use natural language to find the perfect candidates. Try describing what you're looking for!
        </p>
      </div>

      {/* Search Input */}
      <div className="mb-6">
        <div className="relative">
          <FiSearch className="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 text-xl" />
          <input
            type="text"
            value={query}
            onChange={(e) => setQuery(e.target.value)}
            onKeyPress={(e) => e.key === 'Enter' && handleSearch(1)}
            placeholder="e.g., Python developer from ENSAM with ML experience..."
            className="w-full pl-12 pr-24 py-4 text-lg border-2 border-gray-300 rounded-xl focus:border-purple-500 focus:ring-4 focus:ring-purple-100 transition-all"
            disabled={loading}
          />
          {query && (
            <button
              onClick={handleClearSearch}
              className="absolute right-20 top-1/2 -translate-y-1/2 p-2 text-gray-400 hover:text-gray-600"
            >
              <FiX />
            </button>
          )}
          <button
            onClick={() => handleSearch(1)}
            disabled={loading || !query.trim()}
            className="absolute right-2 top-1/2 -translate-y-1/2 px-6 py-2 bg-gradient-to-r from-purple-600 to-pink-600 text-white rounded-lg hover:shadow-lg disabled:opacity-50 disabled:cursor-not-allowed transition-all"
          >
            {loading ? 'Searching...' : 'Search'}
          </button>
        </div>

        {/* Example Queries */}
        <div className="mt-4">
          <p className="text-sm text-gray-500 mb-2">Try these examples:</p>
          <div className="flex flex-wrap gap-2">
            {exampleQueries.map((example, idx) => (
              <button
                key={idx}
                onClick={() => handleExampleClick(example)}
                className="px-3 py-1.5 text-sm bg-gray-100 hover:bg-gray-200 text-gray-700 rounded-full transition-colors"
              >
                {example}
              </button>
            ))}
          </div>
        </div>
      </div>

      {/* Error Message */}
      {error && (
        <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg text-red-700">
          {error}
        </div>
      )}

      {/* Success Message */}
      {successMessage && (
        <div className="mb-6 p-4 bg-green-50 border border-green-200 rounded-lg text-green-700">
          {successMessage}
        </div>
      )}

      {/* Search Results */}
      {searchResult && (
        <div>
          {/* Extracted Criteria */}
          <div className="mb-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
            <h3 className="font-semibold text-blue-900 mb-2">Extracted Search Criteria:</h3>
            <div className="flex flex-wrap gap-2">
              {searchResult.extracted_criteria.skills.length > 0 && (
                <div className="flex items-center gap-1">
                  <span className="text-sm font-medium text-blue-700">Skills:</span>
                  {searchResult.extracted_criteria.skills.map((skill, idx) => (
                    <span
                      key={idx}
                      className="px-2 py-1 bg-blue-100 text-blue-700 text-xs rounded-full"
                    >
                      {skill}
                    </span>
                  ))}
                </div>
              )}
              {searchResult.extracted_criteria.university && (
                <span className="px-2 py-1 bg-green-100 text-green-700 text-xs rounded-full">
                  🎓 {searchResult.extracted_criteria.university}
                </span>
              )}
              {searchResult.extracted_criteria.degree && (
                <span className="px-2 py-1 bg-purple-100 text-purple-700 text-xs rounded-full">
                  📜 {searchResult.extracted_criteria.degree}
                </span>
              )}
              {searchResult.extracted_criteria.field_of_study && (
                <span className="px-2 py-1 bg-pink-100 text-pink-700 text-xs rounded-full">
                  📚 {searchResult.extracted_criteria.field_of_study}
                </span>
              )}
              {searchResult.extracted_criteria.experience_level && (
                <span className="px-2 py-1 bg-orange-100 text-orange-700 text-xs rounded-full">
                  💼 {searchResult.extracted_criteria.experience_level}
                </span>
              )}
            </div>
          </div>

          {/* Results Header */}
          <div className="mb-4 flex items-center justify-between">
            <h3 className="text-xl font-semibold text-gray-800">
              Found {searchResult.total} candidate{searchResult.total !== 1 ? 's' : ''}
            </h3>
            <p className="text-sm text-gray-500">
              Page {searchResult.page} of {searchResult.total_pages}
            </p>
          </div>

          {/* Results List */}
          {searchResult.results.length === 0 ? (
            <div className="text-center py-12 bg-gray-50 rounded-lg">
              <p className="text-gray-600 text-lg">No matching candidates found</p>
              <p className="text-gray-500 text-sm mt-2">Try adjusting your search criteria</p>
            </div>
          ) : (
            <div className="space-y-4">
              {searchResult.results.map((result) => (
                <div
                  key={result.id}
                  className="bg-white border-2 border-gray-200 rounded-xl p-6 hover:shadow-xl transition-all"
                >
                  <div className="flex items-start justify-between mb-4">
                    {/* Profile Header */}
                    <div className="flex items-start gap-4 flex-1">
                      <div className="w-16 h-16 rounded-full bg-gradient-to-br from-purple-400 to-pink-400 flex items-center justify-center text-white text-2xl font-bold flex-shrink-0">
                        {result.profile.profile_image ? (
                          <img
                            src={result.profile.profile_image}
                            alt={result.profile.full_name}
                            className="w-full h-full rounded-full object-cover"
                          />
                        ) : (
                          result.profile.full_name.charAt(0).toUpperCase()
                        )}
                      </div>
                      <div className="flex-1">
                        <h4 className="text-xl font-bold text-gray-900 mb-1">
                          {result.profile.full_name}
                        </h4>
                        <p className="text-gray-600 text-sm mb-2">{result.profile.email}</p>
                        {result.profile.bio && (
                          <p className="text-gray-700 text-sm line-clamp-2">{result.profile.bio}</p>
                        )}
                      </div>
                    </div>

                    {/* AI Score Badge */}
                    <div
                      className={`flex flex-col items-center px-4 py-2 rounded-lg border-2 ${getScoreColor(
                        result.score
                      )}`}
                    >
                      <div className="text-3xl font-bold">{result.score}</div>
                      <div className="text-xs font-semibold">{getScoreLabel(result.score)}</div>
                    </div>
                  </div>

                  {/* AI Reason */}
                  <div className="mb-4 p-3 bg-purple-50 border border-purple-200 rounded-lg">
                    <p className="text-sm text-purple-900">
                      <span className="font-semibold">AI Analysis:</span> {result.reason}
                    </p>
                  </div>

                  {/* Matching Criteria */}
                  {result.matching_criteria.length > 0 && (
                    <div className="mb-4">
                      <p className="text-xs font-semibold text-gray-600 mb-1">Matching Criteria:</p>
                      <div className="flex flex-wrap gap-1">
                        {result.matching_criteria.map((match, idx) => (
                          <span
                            key={idx}
                            className="px-2 py-0.5 bg-green-100 text-green-700 text-xs rounded-full"
                          >
                            ✓ {match}
                          </span>
                        ))}
                      </div>
                    </div>
                  )}

                  {/* Profile Details Grid */}
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                    {/* Skills */}
                    {result.profile.skills.length > 0 && (
                      <div>
                        <div className="flex items-center gap-1 text-sm font-semibold text-gray-700 mb-1">
                          <FiAward className="text-purple-600" />
                          Skills
                        </div>
                        <div className="flex flex-wrap gap-1">
                          {result.profile.skills.slice(0, 5).map((skill) => (
                            <span
                              key={skill.id}
                              className="px-2 py-0.5 bg-gray-100 text-gray-700 text-xs rounded"
                            >
                              {skill.name}
                            </span>
                          ))}
                          {result.profile.skills.length > 5 && (
                            <span className="px-2 py-0.5 text-gray-500 text-xs">
                              +{result.profile.skills.length - 5} more
                            </span>
                          )}
                        </div>
                      </div>
                    )}

                    {/* Education */}
                    {result.profile.educations.length > 0 && (
                      <div>
                        <div className="flex items-center gap-1 text-sm font-semibold text-gray-700 mb-1">
                          <FiBookOpen className="text-blue-600" />
                          Education
                        </div>
                        <p className="text-sm text-gray-700">
                          {result.profile.educations[0].degree_type} in{' '}
                          {result.profile.educations[0].field_of_study}
                        </p>
                        <p className="text-xs text-gray-500">{result.profile.educations[0].university}</p>
                      </div>
                    )}

                    {/* Experience */}
                    {result.profile.experiences.length > 0 && (
                      <div>
                        <div className="flex items-center gap-1 text-sm font-semibold text-gray-700 mb-1">
                          <FiBriefcase className="text-green-600" />
                          Latest Experience
                        </div>
                        <p className="text-sm text-gray-700">{result.profile.experiences[0].job_title}</p>
                        <p className="text-xs text-gray-500">
                          {result.profile.experiences[0].company_name}
                        </p>
                      </div>
                    )}
                  </div>

                  {/* Action Buttons */}
                  <div className="grid grid-cols-2 gap-3">
                    <button
                      onClick={() => handleAddToShortlist(result.id, result.profile.full_name)}
                      disabled={shortlistingId === result.id}
                      className="py-2 bg-gradient-to-r from-green-600 to-emerald-600 text-white rounded-lg hover:shadow-lg transition-all font-medium disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      {shortlistingId === result.id ? (
                        <span>Adding...</span>
                      ) : (
                        <>
                          <FiAward className="inline mr-2" />
                          Add to Shortlist
                        </>
                      )}
                    </button>
                    <button
                      onClick={() => handleViewProfile(result.id)}
                      className="py-2 bg-gradient-to-r from-purple-600 to-pink-600 text-white rounded-lg hover:shadow-lg transition-all font-medium"
                    >
                      <FiUser className="inline mr-2" />
                      View Profile
                    </button>
                  </div>
                </div>
              ))}
            </div>
          )}

          {/* Pagination */}
          {searchResult.total_pages > 1 && (
            <div className="mt-6 flex items-center justify-center gap-2">
              <button
                onClick={() => handleSearch(currentPage - 1)}
                disabled={currentPage === 1 || loading}
                className="px-4 py-2 bg-gray-200 text-gray-700 rounded-lg disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-300 transition-colors"
              >
                Previous
              </button>
              <span className="px-4 py-2 text-gray-700">
                Page {currentPage} of {searchResult.total_pages}
              </span>
              <button
                onClick={() => handleSearch(currentPage + 1)}
                disabled={currentPage === searchResult.total_pages || loading}
                className="px-4 py-2 bg-gray-200 text-gray-700 rounded-lg disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-300 transition-colors"
              >
                Next
              </button>
            </div>
          )}
        </div>
      )}
    </div>
  );
};

export default SmartSearch;
