import React, { useState, useEffect } from 'react';
import { 
  FiStar, 
  FiMail, 
  FiPhone, 
  FiBook,
  FiEye,
  FiTrash2,
  FiSearch,
  FiFilter,
  FiDownload,
  FiX,
  FiExternalLink,
  FiCalendar,
  FiUsers,
  FiTrendingUp
} from 'react-icons/fi';
import apiClient from '../apiClient';

interface ShortlistCandidate {
  student_id: string;
  full_name: string;
  email: string;
  phone?: string;
  photo_url?: string;
  degree?: string;
  university?: string;
  graduation_year?: number;
  location?: string;
  skills: Array<{ id: string; name: string; level?: string }>;
  normalized_skills?: Array<{ name: string; aliases: string[] }>;
  normalized_skill_names?: string[];
  educations: Array<{
    id: string;
    school: string;
    degree?: string;
    field?: string;
    start_year?: number;
    end_year?: number;
    description?: string;
  }>;
  private_note?: string;
  created_at: string;
}

const ShortlistManager: React.FC = () => {
  const [shortlist, setShortlist] = useState<ShortlistCandidate[]>([]);
  const [filteredShortlist, setFilteredShortlist] = useState<ShortlistCandidate[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterSkill, setFilterSkill] = useState('');
  const [selectedCandidate, setSelectedCandidate] = useState<ShortlistCandidate | null>(null);
  const [showModal, setShowModal] = useState(false);
  const [showExportMenu, setShowExportMenu] = useState(false);
  const [exporting, setExporting] = useState(false);

  useEffect(() => {
    fetchShortlist();
  }, []);

  useEffect(() => {
    const filterCandidates = () => {
      let filtered = shortlist;

      const matchesSkill = (candidate: ShortlistCandidate, term: string) => {
        const names = getSkillNames(candidate);
        return names.some(name => name.toLowerCase().includes(term.toLowerCase()));
      };

      if (searchTerm) {
        filtered = filtered.filter(candidate =>
          candidate.full_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
          candidate.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
          candidate.university?.toLowerCase().includes(searchTerm.toLowerCase()) ||
          candidate.degree?.toLowerCase().includes(searchTerm.toLowerCase()) ||
          matchesSkill(candidate, searchTerm)
        );
      }

      if (filterSkill) {
        filtered = filtered.filter(candidate => matchesSkill(candidate, filterSkill));
      }

      setFilteredShortlist(filtered);
    };

    filterCandidates();
  }, [shortlist, searchTerm, filterSkill]);

  // Close export menu when clicking outside or pressing ESC
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      const target = event.target as HTMLElement;
      if (showExportMenu && !target.closest('.export-menu-container')) {
        setShowExportMenu(false);
      }
    };

    const handleKeyDown = (event: KeyboardEvent) => {
      if (event.key === 'Escape' && showExportMenu) {
        setShowExportMenu(false);
      }
    };

    if (showExportMenu) {
      document.addEventListener('mousedown', handleClickOutside);
      document.addEventListener('keydown', handleKeyDown);
      return () => {
        document.removeEventListener('mousedown', handleClickOutside);
        document.removeEventListener('keydown', handleKeyDown);
      };
    }
  }, [showExportMenu]);

  const fetchShortlist = async () => {
    try {
      setLoading(true);
      setError(null);
      const token = localStorage.getItem('token');
      if (!token) {
        throw new Error('No authentication token found');
      }

      console.log('🔍 Fetching shortlist with token:', token?.substring(0, 20) + '...');
      const response = await apiClient.get('/api/shortlist', {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      console.log('📥 Full shortlist response:', response);
      console.log('📄 Response data:', response.data);
      console.log('📋 Shortlist array:', response.data.shortlist);
      
      const shortlistData = response.data.shortlist || response.data || [];
      console.log('✅ Setting shortlist data:', shortlistData);
      setShortlist(shortlistData);
    } catch (err: unknown) {
      console.error('❌ Error fetching shortlist:', err);
      if (err instanceof Error) {
        console.error('Error message:', err.message);
        console.error('Error stack:', err.stack);
      }
      const errorMessage = err instanceof Error ? err.message : 'Failed to fetch shortlist';
      setError(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const removeFromShortlist = async (candidateId: string) => {
    try {
      const token = localStorage.getItem('token');
      if (!token) {
        throw new Error('No authentication token found');
      }

      await apiClient.delete(`/api/shortlist/${candidateId}`, {
        headers: { Authorization: `Bearer ${token}` }
      });

      setShortlist(shortlist.filter(item => item.student_id !== candidateId));
    } catch (err: unknown) {
      console.error('Error removing from shortlist:', err);
      const errorMessage = err instanceof Error ? err.message : 'Failed to remove candidate';
      setError(errorMessage);
    }
  };

  const getSkillNames = (candidate?: ShortlistCandidate | null) => {
    if (!candidate) return [];
    if (candidate.normalized_skill_names && candidate.normalized_skill_names.length > 0) {
      return candidate.normalized_skill_names;
    }
    if (candidate.normalized_skills && candidate.normalized_skills.length > 0) {
      return candidate.normalized_skills.map(s => s.name);
    }
    return candidate.skills.map(skill => skill.name);
  };

  // const getSkillsWithAliases = (candidate?: ShortlistCandidate | null) => {
  //   if (!candidate) return [];
  //   if (candidate.normalized_skills && candidate.normalized_skills.length > 0) {
  //     return candidate.normalized_skills;
  //   }
  //   return [];
  // };

  const getAllSkills = () => {
    const skills = new Set<string>();
    shortlist.forEach(candidate => {
      getSkillNames(candidate).forEach(name => skills.add(name));
    });
    return Array.from(skills).sort((a, b) => a.localeCompare(b));
  };

  const getAliasesTooltip = (skillName: string, candidate: ShortlistCandidate) => {
    const skillWithAliases = candidate.normalized_skills?.find(s => s.name === skillName);
    if (skillWithAliases && skillWithAliases.aliases && skillWithAliases.aliases.length > 0) {
      return `${skillName} (${skillWithAliases.aliases.join(', ')})`;
    }
    return skillName;
  };

  const openModal = (candidate: ShortlistCandidate) => {
    setSelectedCandidate(candidate);
    setShowModal(true);
  };

  const closeModal = () => {
    setSelectedCandidate(null);
    setShowModal(false);
  };

  const exportToCSV = async () => {
    try {
      setExporting(true);
      const token = localStorage.getItem('token');
      if (!token) {
        throw new Error('No authentication token found');
      }

      const response = await apiClient.get('/api/shortlist/export/csv', {
        headers: { Authorization: `Bearer ${token}` },
        responseType: 'blob'
      });

      // Create blob and download
      const blob = new Blob([response.data], { type: 'text/csv;charset=utf-8;' });
      const link = document.createElement('a');
      const url = URL.createObjectURL(blob);
      link.setAttribute('href', url);
      link.setAttribute('download', `shortlist_${new Date().toISOString().split('T')[0]}.csv`);
      link.style.visibility = 'hidden';
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);

      setShowExportMenu(false);
    } catch (err: unknown) {
      console.error('Error exporting CSV:', err);
      const errorMessage = err instanceof Error ? err.message : 'Failed to export CSV';
      setError(errorMessage);
    } finally {
      setExporting(false);
    }
  };

  const exportToPDF = async () => {
    try {
      setExporting(true);
      const token = localStorage.getItem('token');
      if (!token) {
        throw new Error('No authentication token found');
      }

      const response = await apiClient.get('/api/shortlist/export/pdf', {
        headers: { Authorization: `Bearer ${token}` },
        responseType: 'blob'
      });

      // Create blob and download
      const blob = new Blob([response.data], { type: 'text/html;charset=utf-8;' });
      const link = document.createElement('a');
      const url = URL.createObjectURL(blob);
      link.setAttribute('href', url);
      link.setAttribute('download', `shortlist_${new Date().toISOString().split('T')[0]}.html`);
      link.style.visibility = 'hidden';
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);

      setShowExportMenu(false);
    } catch (err: unknown) {
      console.error('Error exporting PDF:', err);
      const errorMessage = err instanceof Error ? err.message : 'Failed to export PDF';
      setError(errorMessage);
    } finally {
      setExporting(false);
    }
  };


  if (loading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-16 w-16 border-b-4 border-blue-600 mx-auto mb-4"></div>
          <p className="text-xl font-semibold text-gray-700">Loading your shortlist...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 flex items-center justify-center">
        <div className="bg-white/80 backdrop-blur-sm rounded-2xl p-8 shadow-xl border border-white/50 text-center max-w-md">
          <div className="text-orange-500 text-6xl mb-4">⚠️</div>
          <h2 className="text-2xl font-bold text-gray-900 mb-2">Error Loading Shortlist</h2>
          <p className="text-gray-600 mb-4">{error}</p>
          <button
            onClick={fetchShortlist}
            className="bg-gradient-to-r from-blue-600 to-purple-600 text-white px-6 py-3 rounded-xl hover:from-blue-700 hover:to-purple-700 transition-all duration-300 font-semibold"
          >
            Try Again
          </button>
        </div>
      </div>
    );
  }

  // Debug info
  console.log('🔍 Current shortlist state:', shortlist);
  console.log('🔍 Current filteredShortlist state:', filteredShortlist);

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100">
      {/* CSS Animation Styles */}
      <style>{`
        @keyframes fadeInUp {
          from {
            opacity: 0;
            transform: translateY(30px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }
        
        @keyframes floating {
          0%, 100% { transform: translateY(0px); }
          50% { transform: translateY(-10px); }
        }
        
        .floating-animation {
          animation: floating 3s ease-in-out infinite;
        }
      `}</style>
      
      {/* Enhanced Header */}
      <div className="relative overflow-hidden bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-700 text-white">
        <div className="absolute inset-0 bg-black/10"></div>
        <div className="absolute inset-0 opacity-20">
          <div 
            className="w-full h-full"
            style={{
              backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23ffffff' fill-opacity='0.05'%3E%3Ccircle cx='30' cy='30' r='4'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`
            }}
          ></div>
        </div>
        
        <div className="relative max-w-7xl mx-auto px-6 py-16">
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <div className="relative">
                <div className="w-20 h-20 bg-white/20 backdrop-blur-sm rounded-2xl flex items-center justify-center mr-6 border border-white/30">
                  <FiStar className="text-white text-3xl" />
                </div>
                <div className="absolute -inset-1 bg-gradient-to-r from-white/20 to-white/5 rounded-2xl blur-lg"></div>
              </div>
              <div>
                <h1 className="text-5xl font-bold mb-3 bg-gradient-to-r from-white to-blue-100 bg-clip-text text-transparent">
                  Talent Shortlist
                </h1>
                <p className="text-xl text-blue-100 flex items-center mb-2">
                  <FiUsers className="mr-3 text-lg" />
                  {filteredShortlist.length} Candidates • {getAllSkills().length} Skills
                </p>
                <p className="text-blue-200">
                  Manage your top candidates and build exceptional teams ⭐
                </p>
              </div>
            </div>
            
            {/* Stats */}
            <div className="text-right">
              <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
                <div className="text-3xl font-bold">{shortlist.length}</div>
                <div className="text-blue-100">Total Candidates</div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-7xl mx-auto px-6 py-12">
        {/* Search & Filter Bar with Export Button */}
        <div className="flex flex-col lg:flex-row gap-6 mb-8">
          {/* Enhanced Search & Filter Bar */}
          <div className="flex-1 relative bg-white/70 backdrop-blur-sm border border-white/50 rounded-2xl p-8 shadow-lg">
            <div className="absolute top-0 right-0 w-40 h-40 bg-gradient-to-br from-blue-500/5 to-purple-500/5 rounded-full blur-2xl transform translate-x-20 -translate-y-20"></div>
            <div className="relative">
              <div className="flex flex-col lg:flex-row gap-6">
                {/* Search */}
                <div className="flex-1">
                  <div className="relative">
                    <input
                      type="text"
                      placeholder="Search by name, email, university, degree, or skills..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="w-full p-4 bg-white/80 backdrop-blur-sm border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all duration-300 pl-12 text-lg"
                    />
                    <FiSearch className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400 text-xl" />
                  </div>
                </div>
                
                {/* Skill Filter */}
                <div className="lg:w-80">
                  <div className="relative">
                    <select
                      value={filterSkill}
                      onChange={(e) => setFilterSkill(e.target.value)}
                      className="w-full p-4 bg-white/80 backdrop-blur-sm border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all duration-300 text-lg appearance-none"
                    >
                      <option value="">All Skills</option>
                      {getAllSkills().map(skill => (
                        <option key={skill} value={skill}>{skill}</option>
                      ))}
                    </select>
                    <FiFilter className="absolute right-4 top-1/2 transform -translate-y-1/2 text-gray-400 text-xl pointer-events-none" />
                  </div>
                </div>
              </div>
            </div>
          </div>
          
          {/* Export Button with Dropdown - Positioned Outside */}
          <div className="relative lg:w-auto export-menu-container">
            <button 
              onClick={() => setShowExportMenu(!showExportMenu)}
              disabled={exporting || shortlist.length === 0}
              className="bg-gradient-to-r from-green-600 via-emerald-500 to-green-600 text-white px-8 py-4 rounded-xl hover:from-green-700 hover:via-emerald-600 hover:to-green-700 transition-all duration-300 font-semibold shadow-lg hover:shadow-2xl flex items-center space-x-3 w-full lg:w-auto disabled:opacity-50 disabled:cursor-not-allowed h-16 active:scale-95 hover:scale-105 transform"
            >
              <FiDownload className={`text-xl transition-transform ${exporting ? 'animate-bounce' : 'group-hover:translate-y-0.5'}`} />
              <span>{exporting ? 'Exporting...' : 'Export'}</span>
            </button>
            
            {showExportMenu && (
              <div className="absolute left-1/2 -translate-x-1/2 top-full mt-1 w-64 bg-white rounded-2xl shadow-2xl border border-gray-200 overflow-hidden z-50 min-w-max animate-in fade-in slide-in-from-top-2 duration-200">
                {/* Header */}
                <div className="bg-gradient-to-r from-green-50 to-emerald-50 px-5 py-3 border-b border-gray-100">
                  <p className="text-sm font-semibold text-gray-700">Choose Format</p>
                </div>

                {/* CSV Export Option */}
                <button
                  onClick={exportToCSV}
                  disabled={exporting}
                  className="w-full text-left px-5 py-4 hover:bg-blue-50 flex items-start space-x-4 text-gray-700 hover:text-blue-700 transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed border-b border-gray-100 group"
                >
                  <div className="flex-shrink-0 mt-1">
                    <div className="w-11 h-11 bg-blue-100 group-hover:bg-blue-200 rounded-lg flex items-center justify-center transition-colors shadow-sm">
                      <span className="text-lg">📊</span>
                    </div>
                  </div>
                  <div className="flex-1">
                    <div className="font-semibold text-gray-900 group-hover:text-blue-700">CSV Format</div>
                    <div className="text-xs text-gray-500 mt-1">Excel spreadsheet</div>
                  </div>
                  <div className="flex-shrink-0 text-gray-400 group-hover:text-blue-400 transition-colors">
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                    </svg>
                  </div>
                </button>

                {/* HTML Export Option */}
                <button
                  onClick={exportToPDF}
                  disabled={exporting}
                  className="w-full text-left px-5 py-4 hover:bg-purple-50 flex items-start space-x-4 text-gray-700 hover:text-purple-700 transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed group"
                >
                  <div className="flex-shrink-0 mt-1">
                    <div className="w-11 h-11 bg-purple-100 group-hover:bg-purple-200 rounded-lg flex items-center justify-center transition-colors shadow-sm">
                      <span className="text-lg">📄</span>
                    </div>
                  </div>
                  <div className="flex-1">
                    <div className="font-semibold text-gray-900 group-hover:text-purple-700">HTML Format</div>
                    <div className="text-xs text-gray-500 mt-1">Print-friendly document</div>
                  </div>
                  <div className="flex-shrink-0 text-gray-400 group-hover:text-purple-400 transition-colors">
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                    </svg>
                  </div>
                </button>
              </div>
            )}
          </div>
        </div>

        {/* Enhanced Results Summary */}
        {filteredShortlist.length > 0 && (
          <div className="mb-8">
            <div className="bg-white/70 backdrop-blur-sm rounded-2xl p-6 border border-white/50 shadow-lg">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <div className="w-12 h-12 bg-gradient-to-br from-green-500 to-emerald-600 rounded-xl flex items-center justify-center">
                    <FiUsers className="text-white text-xl" />
                  </div>
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900">
                      {filteredShortlist.length === shortlist.length 
                        ? `Showing all ${shortlist.length} candidates`
                        : `${filteredShortlist.length} of ${shortlist.length} candidates`
                      }
                    </h3>
                    <p className="text-gray-600">
                      {searchTerm || filterSkill 
                        ? 'Filtered results based on your search criteria'
                        : 'Your complete talent pipeline'
                      }
                    </p>
                  </div>
                </div>
                
                {(searchTerm || filterSkill) && (
                  <button
                    onClick={() => {
                      setSearchTerm('');
                      setFilterSkill('');
                    }}
                    className="bg-gray-100 hover:bg-gray-200 text-gray-700 px-4 py-2 rounded-lg transition-colors duration-200 flex items-center space-x-2"
                  >
                    <FiX className="text-sm" />
                    <span>Clear filters</span>
                  </button>
                )}
              </div>
            </div>
          </div>
        )}

        {/* Candidates Grid */}
        {filteredShortlist.length === 0 ? (
          <div className="text-center py-20">
            <div className="relative bg-white/70 backdrop-blur-sm rounded-3xl p-16 border border-white/50 shadow-xl max-w-2xl mx-auto">
              {/* Floating elements */}
              <div className="absolute top-6 left-6 w-4 h-4 bg-blue-400 rounded-full animate-pulse"></div>
              <div className="absolute top-12 right-8 w-3 h-3 bg-purple-400 rounded-full animate-pulse delay-500"></div>
              <div className="absolute bottom-8 left-12 w-2 h-2 bg-indigo-400 rounded-full animate-pulse delay-1000"></div>
              
              <div className="relative">
                {shortlist.length === 0 ? (
                  <>
                    <div className="mb-8">
                      <div className="w-32 h-32 bg-gradient-to-br from-blue-100 to-purple-100 rounded-full flex items-center justify-center mx-auto mb-6 border-8 border-white shadow-xl">
                        <FiStar className="text-5xl text-blue-600" />
                      </div>
                      <h3 className="text-3xl font-bold text-gray-900 mb-4">
                        Start Building Your Dream Team
                      </h3>
                      <p className="text-gray-600 text-lg mb-8 leading-relaxed">
                        Your talent shortlist is empty. Start scanning QR codes at career events,<br />
                        networking sessions, or job fairs to discover exceptional candidates.
                      </p>
                    </div>
                    
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-10">
                      <div className="text-center p-6 bg-blue-50/80 rounded-2xl border border-blue-100">
                        <div className="w-16 h-16 bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl flex items-center justify-center mx-auto mb-4">
                          <span className="text-white text-2xl">📱</span>
                        </div>
                        <h4 className="font-semibold text-gray-900 mb-2">Scan QR Codes</h4>
                        <p className="text-gray-600 text-sm">Find candidates at events</p>
                      </div>
                      
                      <div className="text-center p-6 bg-purple-50/80 rounded-2xl border border-purple-100">
                        <div className="w-16 h-16 bg-gradient-to-br from-purple-500 to-purple-600 rounded-xl flex items-center justify-center mx-auto mb-4">
                          <span className="text-white text-2xl">⭐</span>
                        </div>
                        <h4 className="font-semibold text-gray-900 mb-2">Build Shortlist</h4>
                        <p className="text-gray-600 text-sm">Curate top talent</p>
                      </div>
                      
                      <div className="text-center p-6 bg-green-50/80 rounded-2xl border border-green-100">
                        <div className="w-16 h-16 bg-gradient-to-br from-green-500 to-green-600 rounded-xl flex items-center justify-center mx-auto mb-4">
                          <span className="text-white text-2xl">🚀</span>
                        </div>
                        <h4 className="font-semibold text-gray-900 mb-2">Hire Faster</h4>
                        <p className="text-gray-600 text-sm">Connect with candidates</p>
                      </div>
                    </div>
                    
                    <button className="bg-gradient-to-r from-blue-600 to-purple-600 text-white px-10 py-4 rounded-2xl hover:from-blue-700 hover:to-purple-700 transition-all duration-300 font-semibold text-lg shadow-xl hover:shadow-2xl transform hover:-translate-y-1">
                      <FiSearch className="inline mr-3 text-xl" />
                      Start Discovering Talent
                    </button>
                  </>
                ) : (
                  <>
                    <div className="mb-8">
                      <div className="w-24 h-24 bg-gradient-to-br from-gray-100 to-gray-200 rounded-full flex items-center justify-center mx-auto mb-6">
                        <FiSearch className="text-4xl text-gray-400" />
                      </div>
                      <h3 className="text-2xl font-bold text-gray-900 mb-4">
                        No matches found
                      </h3>
                      <p className="text-gray-600 text-lg mb-6">
                        Try adjusting your search terms or filters to find candidates
                      </p>
                    </div>
                    
                    <div className="flex flex-col sm:flex-row gap-4 justify-center">
                      <button
                        onClick={() => setSearchTerm('')}
                        className="bg-blue-600 text-white px-6 py-3 rounded-xl hover:bg-blue-700 transition-colors font-semibold"
                      >
                        Clear search
                      </button>
                      <button
                        onClick={() => setFilterSkill('')}
                        className="bg-purple-600 text-white px-6 py-3 rounded-xl hover:bg-purple-700 transition-colors font-semibold"
                      >
                        Clear filters
                      </button>
                    </div>
                  </>
                )}
              </div>
            </div>
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-10">
            {filteredShortlist.map((candidate, index) => (
              <div
                key={candidate.student_id}
                className="group relative overflow-hidden bg-white/70 backdrop-blur-sm border border-white/50 rounded-3xl p-10 hover:bg-white/80 hover:shadow-2xl hover:border-white/70 transition-all duration-500 cursor-pointer transform hover:-translate-y-2"
                onClick={() => openModal(candidate)}
                style={{
                  animationDelay: `${index * 100}ms`,
                  animation: 'fadeInUp 0.6s ease-out forwards'
                }}
              >
                {/* Floating gradient orb */}
                <div className="absolute top-0 right-0 w-40 h-40 bg-gradient-to-br from-blue-500/10 via-purple-500/10 to-indigo-500/10 rounded-full blur-2xl transform translate-x-20 -translate-y-20 group-hover:scale-150 group-hover:translate-x-16 group-hover:-translate-y-16 transition-all duration-700"></div>
                
                {/* Hover glow effect */}
                <div className="absolute inset-0 bg-gradient-to-r from-blue-500/5 to-purple-500/5 rounded-3xl opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                
                <div className="relative">
                  {/* Profile Section with enhanced spacing */}
                  <div className="flex items-start space-x-6 mb-10">
                    <div className="relative">
                      {candidate.photo_url ? (
                        <img
                          src={candidate.photo_url}
                          alt={candidate.full_name}
                          className="w-24 h-24 rounded-2xl object-cover border-3 border-white shadow-xl group-hover:shadow-2xl transition-all duration-300"
                        />
                      ) : (
                        <div className="w-24 h-24 bg-gradient-to-br from-blue-500 to-purple-600 rounded-2xl flex items-center justify-center text-white text-2xl font-bold shadow-xl group-hover:shadow-2xl group-hover:scale-105 transition-all duration-300">
                          {candidate.full_name.charAt(0).toUpperCase()}
                        </div>
                      )}
                      {/* Online indicator */}
                      <div className="absolute -bottom-1 -right-1 w-6 h-6 bg-green-500 rounded-full border-3 border-white shadow-lg"></div>
                    </div>
                    
                    <div className="flex-1 min-w-0">
                      <h3 className="text-xl font-bold text-gray-900 mb-3 group-hover:text-blue-600 transition-colors duration-300 truncate">
                        {candidate.full_name}
                      </h3>
                      <p className="text-gray-600 flex items-center mb-3 text-sm">
                        <FiMail className="mr-3 text-blue-500 flex-shrink-0" />
                        <span className="truncate">{candidate.email}</span>
                      </p>
                      {candidate.phone && (
                        <p className="text-gray-600 flex items-center text-sm">
                          <FiPhone className="mr-3 text-green-500 flex-shrink-0" />
                          <span className="truncate">{candidate.phone}</span>
                        </p>
                      )}
                    </div>
                  </div>

                  {/* Education with modern card */}
                  {(candidate.degree || candidate.university) && (
                    <div className="mb-8 p-6 bg-gradient-to-r from-blue-50/80 to-indigo-50/80 rounded-2xl border border-blue-100/50 backdrop-blur-sm">
                      <div className="flex items-center mb-4">
                        <div className="w-8 h-8 bg-gradient-to-r from-blue-500 to-indigo-600 rounded-lg flex items-center justify-center mr-3">
                          <FiBook className="text-white text-sm" />
                        </div>
                        <span className="font-semibold text-blue-900">Education</span>
                      </div>
                      {candidate.degree && (
                        <p className="text-gray-800 font-medium mb-2">{candidate.degree}</p>
                      )}
                      {candidate.university && (
                        <p className="text-gray-600 text-sm mb-2">{candidate.university}</p>
                      )}
                      {candidate.graduation_year && (
                        <p className="text-gray-500 text-xs">Class of {candidate.graduation_year}</p>
                      )}
                    </div>
                  )}

                  {/* Enhanced Skills Section */}
                  {getSkillNames(candidate).length > 0 && (
                    <div className="mb-10">
                      <h4 className="font-semibold text-gray-900 mb-5 flex items-center">
                        <div className="w-8 h-8 bg-gradient-to-r from-purple-500 to-pink-600 rounded-lg flex items-center justify-center mr-3">
                          <FiTrendingUp className="text-white text-sm" />
                        </div>
                        Skills ({getSkillNames(candidate).length})
                      </h4>
                      <div className="flex flex-wrap gap-3">
                        {getSkillNames(candidate).slice(0, 4).map((skill, index) => (
                          <span
                            key={index}
                            title={getAliasesTooltip(skill, candidate)}
                            className="bg-gradient-to-r from-blue-100 to-purple-100 text-blue-800 px-4 py-2 rounded-full text-sm font-medium border border-blue-200/50 backdrop-blur-sm hover:from-blue-200 hover:to-purple-200 transition-all duration-300 transform hover:scale-105 cursor-help"
                          >
                            {skill}
                          </span>
                        ))}
                        {getSkillNames(candidate).length > 4 && (
                          <span className="bg-gradient-to-r from-gray-100 to-gray-200 text-gray-600 px-4 py-2 rounded-full text-sm font-medium border border-gray-200">
                            +{getSkillNames(candidate).length - 4} more
                          </span>
                        )}
                      </div>
                    </div>
                  )}

                  {/* Private Notes with better spacing */}
                  {candidate.private_note && (
                    <div className="mb-10 p-6 bg-gradient-to-r from-amber-50/80 to-yellow-50/80 rounded-2xl border border-amber-100/50 backdrop-blur-sm">
                      <h4 className="font-semibold text-amber-900 mb-4 flex items-center">
                        <div className="w-8 h-8 bg-gradient-to-r from-amber-500 to-orange-600 rounded-lg flex items-center justify-center mr-3">
                          <span className="text-white text-sm">📝</span>
                        </div>
                        Private Note
                      </h4>
                      <p className="text-amber-800 text-sm italic leading-relaxed">{candidate.private_note}</p>
                    </div>
                  )}

                  {/* Enhanced Action Buttons */}
                  <div className="flex space-x-4 mb-6">
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        window.open(`/p/${candidate.student_id}`, '_blank');
                      }}
                      className="flex-1 bg-gradient-to-r from-blue-600 to-purple-600 text-white py-4 rounded-2xl hover:from-blue-700 hover:to-purple-700 transition-all duration-300 font-semibold flex items-center justify-center space-x-2 shadow-lg hover:shadow-xl transform hover:-translate-y-0.5"
                    >
                      <FiEye className="text-lg" />
                      <span>View Profile</span>
                    </button>
                    <div className="relative group/delete">
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          removeFromShortlist(candidate.student_id);
                        }}
                        className="relative overflow-hidden bg-gradient-to-r from-red-500 to-red-600 text-white px-6 py-4 rounded-2xl hover:from-red-600 hover:to-red-700 transition-all duration-300 shadow-lg hover:shadow-xl transform hover:-translate-y-0.5 hover:scale-105 group-hover/delete:animate-pulse"
                      >
                        {/* Animated background effect */}
                        <div className="absolute inset-0 bg-gradient-to-r from-red-400 to-red-500 opacity-0 group-hover/delete:opacity-100 transition-opacity duration-300"></div>
                        
                        {/* Icon with rotation animation */}
                        <FiTrash2 className="relative text-lg transition-transform duration-300 group-hover/delete:rotate-12 group-hover/delete:scale-110" />
                        
                        {/* Ripple effect */}
                        <div className="absolute inset-0 rounded-2xl bg-white/20 scale-0 group-active/delete:scale-100 transition-transform duration-150"></div>
                      </button>
                    </div>
                  </div>

                  {/* Enhanced date with location if available */}
                  <div className="flex items-center justify-between text-xs text-gray-500 pt-4 border-t border-gray-100">
                    <div className="flex items-center">
                      <FiCalendar className="mr-2" />
                      Added {new Date(candidate.created_at).toLocaleDateString('en-US', { 
                        month: 'short', 
                        day: 'numeric',
                        year: 'numeric'
                      })}
                    </div>
                    {candidate.location && (
                      <div className="flex items-center">
                        <span className="mr-1">📍</span>
                        <span className="truncate max-w-20">{candidate.location}</span>
                      </div>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>

      {/* Enhanced Modal */}
      {showModal && selectedCandidate && (
        <div className="fixed inset-0 bg-black/60 backdrop-blur-sm flex items-center justify-center p-6 z-50 animate-in fade-in duration-300">
          <div className="bg-white/95 backdrop-blur-md rounded-3xl max-w-6xl w-full max-h-[90vh] overflow-y-auto border border-white/50 shadow-2xl animate-in slide-in-from-bottom-4 duration-500">
            {/* Enhanced Modal Header */}
            <div className="relative bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-700 text-white p-12 rounded-t-3xl overflow-hidden">
              {/* Background Pattern */}
              <div className="absolute inset-0 opacity-10">
                <div 
                  className="w-full h-full"
                  style={{
                    backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23ffffff' fill-opacity='0.05'%3E%3Ccircle cx='30' cy='30' r='4'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`
                  }}
                ></div>
              </div>
              
              <button
                onClick={closeModal}
                className="absolute top-8 right-8 w-12 h-12 bg-white/20 backdrop-blur-sm rounded-full flex items-center justify-center hover:bg-white/30 transition-all duration-300 hover:rotate-90 z-10"
              >
                <FiX className="text-xl" />
              </button>
              
              <div className="relative flex items-center justify-between pr-24">
                <div className="flex items-center space-x-10 flex-1">
                  <div className="relative">
                    {selectedCandidate.photo_url ? (
                      <img
                        src={selectedCandidate.photo_url}
                        alt={selectedCandidate.full_name}
                        className="w-36 h-36 rounded-3xl object-cover border-4 border-white/30 shadow-2xl"
                      />
                    ) : (
                      <div className="w-36 h-36 bg-white/20 backdrop-blur-sm rounded-3xl flex items-center justify-center text-white text-4xl font-bold border-4 border-white/30 shadow-2xl">
                        {selectedCandidate.full_name.charAt(0).toUpperCase()}
                      </div>
                    )}
                    {/* Enhanced status indicator */}
                    <div className="absolute -bottom-3 -right-3 w-10 h-10 bg-green-500 rounded-full border-4 border-white shadow-lg flex items-center justify-center">
                      <div className="w-4 h-4 bg-white rounded-full animate-pulse"></div>
                    </div>
                  </div>
                  
                  <div className="flex-1 min-w-0">
                    <h2 className="text-5xl font-bold mb-4 bg-gradient-to-r from-white to-blue-100 bg-clip-text text-transparent">
                      {selectedCandidate.full_name}
                    </h2>
                    <div className="space-y-3">
                      <p className="text-blue-100 text-xl flex items-center">
                        <FiMail className="mr-4 text-2xl flex-shrink-0" />
                        <span className="truncate">{selectedCandidate.email}</span>
                      </p>
                      {selectedCandidate.phone && (
                        <p className="text-blue-100 text-xl flex items-center">
                          <FiPhone className="mr-4 text-2xl flex-shrink-0" />
                          <span className="truncate">{selectedCandidate.phone}</span>
                        </p>
                      )}
                      {selectedCandidate.location && (
                        <p className="text-blue-100 text-xl flex items-center">
                          <span className="mr-4 text-2xl flex-shrink-0">📍</span>
                          <span className="truncate">{selectedCandidate.location}</span>
                        </p>
                      )}
                    </div>
                  </div>
                </div>
                
                {/* Enhanced quick stats - positioned to avoid close button */}
                <div className="flex flex-col space-y-6 ml-8">
                  <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20 min-w-32 text-center">
                    <div className="text-3xl font-bold">{getSkillNames(selectedCandidate).length}</div>
                    <div className="text-blue-100 text-sm">Skills</div>
                  </div>
                  <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20 min-w-32 text-center">
                    <div className="text-3xl font-bold">{selectedCandidate.educations.length}</div>
                    <div className="text-blue-100 text-sm">Education</div>
                  </div>
                </div>
              </div>
            </div>

            {/* Enhanced Modal Content */}
            <div className="p-12">
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
                {/* Enhanced Education Section */}
                <div className="bg-gradient-to-br from-blue-50/80 to-indigo-50/80 rounded-3xl p-10 border border-blue-100/50 backdrop-blur-sm">
                  <h3 className="text-3xl font-bold text-blue-900 mb-8 flex items-center">
                    <div className="w-14 h-14 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-2xl flex items-center justify-center mr-5">
                      <FiBook className="text-white text-2xl" />
                    </div>
                    Education Journey
                  </h3>
                  {selectedCandidate.educations.length > 0 ? (
                    <div className="space-y-8">
                      {selectedCandidate.educations.map((education, index) => (
                        <div key={index} className="bg-white/80 backdrop-blur-sm p-8 rounded-2xl border border-blue-200/50 shadow-lg hover:shadow-xl transition-all duration-300">
                          <h4 className="font-bold text-gray-900 text-xl mb-3">{education.school}</h4>
                          {education.degree && (
                            <p className="text-blue-700 font-semibold mb-2 text-lg">{education.degree}</p>
                          )}
                          {education.field && (
                            <p className="text-gray-600 mb-3 text-base">{education.field}</p>
                          )}
                          {(education.start_year || education.end_year) && (
                            <p className="text-gray-500 text-sm flex items-center mb-3">
                              <FiCalendar className="mr-3" />
                              {education.start_year} - {education.end_year || 'Present'}
                            </p>
                          )}
                          {education.description && (
                            <p className="text-gray-600 text-sm mt-4 italic leading-relaxed bg-gray-50/80 p-4 rounded-xl">{education.description}</p>
                          )}
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="text-center py-16">
                      <div className="text-blue-300 text-8xl mb-6">🎓</div>
                      <p className="text-blue-700 text-xl">No education information available</p>
                    </div>
                  )}
                </div>

                {/* Enhanced Skills Section */}
                <div className="bg-gradient-to-br from-purple-50/80 to-pink-50/80 rounded-3xl p-10 border border-purple-100/50 backdrop-blur-sm">
                  <h3 className="text-3xl font-bold text-purple-900 mb-8 flex items-center">
                    <div className="w-14 h-14 bg-gradient-to-br from-purple-500 to-pink-600 rounded-2xl flex items-center justify-center mr-5">
                      <FiTrendingUp className="text-white text-2xl" />
                    </div>
                    Skills Mastery ({getSkillNames(selectedCandidate).length})
                  </h3>
                  {getSkillNames(selectedCandidate).length > 0 ? (
                    <div className="space-y-6">
                      <div className="grid grid-cols-1 gap-4">
                        {getSkillNames(selectedCandidate).map((skillName, index) => (
                          <div
                            key={index}
                            title={getAliasesTooltip(skillName, selectedCandidate)}
                            className="bg-gradient-to-r from-purple-100/80 to-pink-100/80 backdrop-blur-sm border border-purple-200/50 rounded-2xl p-6 hover:from-purple-200/80 hover:to-pink-200/80 transition-all duration-300 transform hover:scale-105 cursor-help"
                          >
                            <div className="flex items-center justify-between">
                              <span className="font-semibold text-purple-800 text-lg">{skillName}</span>
                              {selectedCandidate.normalized_skills?.find(s => s.name === skillName)?.aliases && selectedCandidate.normalized_skills.find(s => s.name === skillName)!.aliases.length > 0 && (
                                <span className="text-xs text-purple-600 bg-white/50 px-3 py-1 rounded-full">
                                  +{selectedCandidate.normalized_skills.find(s => s.name === skillName)!.aliases.length} alias
                                </span>
                              )}
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  ) : (
                    <div className="text-center py-16">
                      <div className="text-purple-300 text-8xl mb-6">🚀</div>
                      <p className="text-purple-700 text-xl">No skills information available</p>
                    </div>
                  )}
                </div>
              </div>

              {/* Enhanced Notes Section */}
              {selectedCandidate.private_note && (
                <div className="mt-12 bg-gradient-to-br from-amber-50/80 to-orange-50/80 rounded-3xl p-10 border border-amber-100/50 backdrop-blur-sm">
                  <h3 className="text-3xl font-bold text-amber-900 mb-8 flex items-center">
                    <div className="w-14 h-14 bg-gradient-to-br from-amber-500 to-orange-600 rounded-2xl flex items-center justify-center mr-5">
                      <span className="text-white text-2xl">📝</span>
                    </div>
                    Private Notes
                  </h3>
                  <div className="bg-white/80 backdrop-blur-sm p-8 rounded-2xl border border-amber-200/50">
                    <p className="text-amber-800 text-xl leading-relaxed italic">{selectedCandidate.private_note}</p>
                  </div>
                </div>
              )}

              {/* Enhanced Action Buttons */}
              <div className="mt-12 flex flex-col sm:flex-row gap-8">
                <button
                  onClick={() => window.open(`/p/${selectedCandidate.student_id}`, '_blank')}
                  className="flex-1 bg-gradient-to-r from-blue-600 to-purple-600 text-white py-6 rounded-2xl hover:from-blue-700 hover:to-purple-700 transition-all duration-300 font-bold flex items-center justify-center space-x-4 text-xl shadow-xl hover:shadow-2xl transform hover:-translate-y-1"
                >
                  <FiExternalLink className="text-2xl" />
                  <span>View Complete Profile</span>
                </button>
                <div className="relative group">
                  <button
                    onClick={() => {
                      removeFromShortlist(selectedCandidate.student_id);
                      closeModal();
                    }}
                    className="relative overflow-hidden bg-gradient-to-r from-red-500 to-red-600 text-white px-12 py-6 rounded-2xl hover:from-red-600 hover:to-red-700 transition-all duration-300 font-bold flex items-center justify-center space-x-4 shadow-xl hover:shadow-2xl transform hover:-translate-y-1 group-hover:animate-pulse"
                  >
                    {/* Animated background effect */}
                    <div className="absolute inset-0 bg-gradient-to-r from-red-400 to-red-500 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                    
                    {/* Enhanced icon with animations */}
                    <FiTrash2 className="relative text-2xl transition-all duration-300 group-hover:rotate-12 group-hover:scale-110" />
                    <span className="relative transition-all duration-300 group-hover:scale-105 text-xl">Remove from Shortlist</span>
                    
                    {/* Ripple effect */}
                    <div className="absolute inset-0 rounded-2xl bg-white/20 scale-0 group-active:scale-100 transition-transform duration-150"></div>
                  </button>
                  
                  {/* Confirmation tooltip */}
                  <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-4 opacity-0 group-hover:opacity-100 transition-all duration-300 pointer-events-none z-50">
                    <div className="bg-red-700 text-white text-sm font-medium px-6 py-3 rounded-xl shadow-xl whitespace-nowrap relative border border-red-600">
                      ⚠️ This will permanently remove the candidate
                      {/* Tooltip arrow */}
                      <div className="absolute top-full left-1/2 transform -translate-x-1/2 w-0 h-0 border-l-4 border-r-4 border-t-4 border-transparent border-t-red-700"></div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default ShortlistManager;
