import React from 'react';
import { BrowserRouter as Router, Routes, Route, Navigate, useLocation } from 'react-router-dom';
import Register from './pages/Register';
import Login from './pages/Login';
import Dashboard from './pages/Dashboard';
import StudentDashboard from './pages/StudentDashboard';
import RecruiterDashboard from './pages/RecruiterDashboard';
import Profile from './pages/Profile';
import Shortlist from './pages/Shortlist';
import RegisterRole from './pages/RegisterRole';
import ScanProfile from './pages/ScanProfile';
import PublicProfile from './pages/PublicProfile';
import QRScanner from './pages/QRScanner';
import MyCard from './pages/MyCard';
import Navbar from './components/Navbar';

// Helper: get user from localStorage
const getUser = () => {
  const user = localStorage.getItem('user');
  return user ? JSON.parse(user) : null;
};

// PrivateRoute: only for authenticated users
const PrivateRoute = ({ children }: { children: React.ReactNode }) => {
  const user = getUser();
  const location = useLocation();
  if (!user || !localStorage.getItem('token')) {
    return <Navigate to="/login" state={{ from: location }} replace />;
  }
  return children;
};

// RoleRoute: only for users with a specific role
const RoleRoute = ({ role, children }: { role: string; children: React.ReactNode }) => {
  const user = getUser();
  const location = useLocation();
  if (!user || !localStorage.getItem('token')) {
    return <Navigate to="/login" state={{ from: location }} replace />;
  }
  if (user.role !== role) {
    // Redirect to their dashboard
    return <Navigate to="/dashboard" replace />;
  }
  return children;
};

// AuthRedirect: redirect already-authenticated users away from login/register
const AuthRedirect = ({ children }: { children: React.ReactNode }) => {
  const user = getUser();
  if (user && localStorage.getItem('token')) {
    return <Navigate to="/dashboard" replace />;
  }
  return children;
};

function App() {
  return (
    <Router>
      <Navbar />
      <Routes>
        <Route path="/" element={<AuthRedirect><Login /></AuthRedirect>} />
        <Route path="/login" element={<AuthRedirect><Login /></AuthRedirect>} />
        <Route path="/register" element={<AuthRedirect><RegisterRole /></AuthRedirect>} />
        <Route path="/register-details" element={<AuthRedirect><Register /></AuthRedirect>} />
        <Route path="/dashboard" element={<PrivateRoute><Dashboard /></PrivateRoute>} />
        <Route path="/profile" element={<PrivateRoute><Profile /></PrivateRoute>} />
        <Route path="/shortlist" element={<RoleRoute role="recruiter"><Shortlist /></RoleRoute>} />
        <Route path="/student-dashboard" element={<RoleRoute role="student"><StudentDashboard /></RoleRoute>} />
        <Route path="/recruiter-dashboard" element={<RoleRoute role="recruiter"><RecruiterDashboard /></RoleRoute>} />
        <Route path="/my-card" element={<RoleRoute role="student"><MyCard /></RoleRoute>} />
        <Route path="/scanner" element={<RoleRoute role="recruiter"><QRScanner /></RoleRoute>} />
        <Route path="/p/:uuid" element={<PublicProfile />} />
        <Route path="/scan-profile/:uuid" element={<PrivateRoute><ScanProfile /></PrivateRoute>} />
        <Route path="*" element={<Navigate to="/login" replace />} />
      </Routes>
    </Router>
  );
}

export default App;
