<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class SmartSearchService
{
    private $groqApiKey;
    private $groqApiUrl = 'https://api.groq.com/openai/v1/chat/completions';

    public function __construct()
    {
        $this->groqApiKey = env('GROQ_API_KEY');
    }

    /**
     * Extract structured search criteria from natural language query using Groq API
     */
    public function extractCriteria(string $query): array
    {
        try {
            $prompt = "Extract search criteria from this query: \"$query\"\n\n" .
                "Return ONLY a JSON object with these fields:\n" .
                "- skills: array of technical skills/technologies mentioned\n" .
                "- university: university name if mentioned (null otherwise)\n" .
                "- degree: degree level if mentioned (null otherwise)\n" .
                "- field_of_study: field of study if mentioned (null otherwise)\n" .
                "- experience_level: experience level if mentioned (junior/mid/senior, null otherwise)\n" .
                "- min_experience_years: minimum years if mentioned (null otherwise)\n\n" .
                "Example query: 'Python developer from ENSAM with machine learning experience'\n" .
                "Example output: {\"skills\":[\"Python\",\"Machine Learning\"],\"university\":\"ENSAM\",\"degree\":null,\"field_of_study\":null,\"experience_level\":null,\"min_experience_years\":null}\n\n" .
                "Return ONLY valid JSON, no other text.";

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->groqApiKey,
                'Content-Type' => 'application/json',
            ])->timeout(30)->post($this->groqApiUrl, [
                'model' => 'llama-3.3-70b-versatile',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a search query parser. Extract structured data from natural language queries. Return only valid JSON.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.1,
                'max_tokens' => 500,
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $content = $data['choices'][0]['message']['content'] ?? '';
                
                // Clean the response - remove markdown code blocks if present
                $content = preg_replace('/```json\s*/', '', $content);
                $content = preg_replace('/```\s*/', '', $content);
                $content = trim($content);
                
                $extracted = json_decode($content, true);
                
                if (json_last_error() === JSON_ERROR_NONE) {
                    Log::info('Smart Search - Criteria extracted', [
                        'query' => $query,
                        'extracted' => $extracted
                    ]);
                    
                    return [
                        'skills' => $extracted['skills'] ?? [],
                        'university' => $extracted['university'] ?? null,
                        'degree' => $extracted['degree'] ?? null,
                        'field_of_study' => $extracted['field_of_study'] ?? null,
                        'experience_level' => $extracted['experience_level'] ?? null,
                        'min_experience_years' => $extracted['min_experience_years'] ?? null,
                    ];
                }
                
                Log::warning('Smart Search - JSON parse error', [
                    'content' => $content,
                    'error' => json_last_error_msg()
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Smart Search - Groq API error', [
                'query' => $query,
                'error' => $e->getMessage()
            ]);
        }

        // Fallback to simple keyword extraction
        return $this->fallbackExtraction($query);
    }

    /**
     * Rank student profiles based on query relevance using Groq API
     */
    public function rankProfiles(string $query, array $profiles): array
    {
        try {
            // Prepare simplified profile data for AI
            $simplifiedProfiles = [];
            foreach ($profiles as $profile) {
                $simplifiedProfiles[] = [
                    'id' => $profile->id,
                    'name' => $profile->full_name ?? 'N/A',
                    'skills' => $profile->skills->pluck('name')->toArray(),
                    'university' => $profile->educations->first()->school ?? null,
                    'degree' => $profile->educations->first()->degree ?? null,
                    'field' => $profile->educations->first()->field ?? null,
                    'experience_years' => $profile->experiences->sum(function($exp) {
                        $start = \Carbon\Carbon::parse($exp->start_date);
                        $end = $exp->end_date ? \Carbon\Carbon::parse($exp->end_date) : now();
                        return $start->diffInYears($end);
                    }),
                ];
            }

            $prompt = "Query: \"$query\"\n\n" .
                "Rank these candidates by relevance (0-100 score):\n" .
                json_encode($simplifiedProfiles, JSON_PRETTY_PRINT) . "\n\n" .
                "Return ONLY a JSON array with objects containing:\n" .
                "- id: candidate ID\n" .
                "- score: relevance score 0-100 (100=perfect match)\n" .
                "- reason: brief explanation (max 100 chars)\n\n" .
                "Example: [{\"id\":\"123\",\"score\":95,\"reason\":\"Strong Python skills, ML background, ENSAM degree\"}]\n\n" .
                "Return ONLY valid JSON array, no other text.";

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->groqApiKey,
                'Content-Type' => 'application/json',
            ])->timeout(45)->post($this->groqApiUrl, [
                'model' => 'llama-3.3-70b-versatile',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a recruitment AI. Rank candidates objectively based on query relevance. Return only valid JSON.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.3,
                'max_tokens' => 2000,
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $content = $data['choices'][0]['message']['content'] ?? '';
                
                // Clean the response
                $content = preg_replace('/```json\s*/', '', $content);
                $content = preg_replace('/```\s*/', '', $content);
                $content = trim($content);
                
                $rankings = json_decode($content, true);
                
                if (json_last_error() === JSON_ERROR_NONE && is_array($rankings)) {
                    Log::info('Smart Search - Rankings generated', [
                        'query' => $query,
                        'count' => count($rankings)
                    ]);
                    
                    return $rankings;
                }
                
                Log::warning('Smart Search - Rankings JSON parse error', [
                    'content' => $content,
                    'error' => json_last_error_msg()
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Smart Search - Ranking error', [
                'query' => $query,
                'error' => $e->getMessage()
            ]);
        }

        // Fallback: return all profiles with default score
        return array_map(function($profile) {
            return [
                'id' => $profile->id,
                'score' => 50,
                'reason' => 'Auto-generated ranking'
            ];
        }, $profiles);
    }

    /**
     * Fallback keyword extraction when AI fails
     */
    private function fallbackExtraction(string $query): array
    {
        $query = strtolower($query);
        
        // Common skills patterns
        $skills = [];
        $skillKeywords = [
            'python', 'javascript', 'java', 'php', 'react', 'vue', 'angular',
            'node', 'laravel', 'django', 'spring', 'sql', 'nosql', 'mongodb',
            'docker', 'kubernetes', 'aws', 'azure', 'gcp', 'machine learning',
            'ml', 'ai', 'data science', 'frontend', 'backend', 'fullstack'
        ];
        
        foreach ($skillKeywords as $skill) {
            if (strpos($query, $skill) !== false) {
                $skills[] = ucfirst($skill);
            }
        }
        
        return [
            'skills' => $skills,
            'university' => null,
            'degree' => null,
            'field_of_study' => null,
            'experience_level' => null,
            'min_experience_years' => null,
        ];
    }
}
