<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Student extends Model
{
    use HasFactory;

    /**
     * The primary key associated with the table.
     */
    protected $primaryKey = 'user_id';

    /**
     * Indicates if the model should use UUIDs for the primary key.
     */
    public $incrementing = false;
    protected $keyType = 'string';

    /**
     * Indicates if the model should be timestamped.
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'user_id',
        'full_name',
        'photo_url',
        'cv_file_url',
        'is_card_active',
        'phone',
        'linkedin',
        'github',
        'summary',
        'university',
        'graduation_year',
        'languages',
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'is_card_active' => 'boolean',
        'languages' => 'array',
    ];

    /**
     * Get the user that owns the student profile.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Calculate profile completion percentage.
     */
    public function getProfileCompletion()
    {
        $fields = [
            !empty($this->summary),
            $this->skills()->count() > 0, // Now using relationship
            !empty($this->university),
            $this->educations()->count() > 0, // Now using relationship
            !empty($this->graduation_year),
            !empty($this->cv_file_url),
        ];
        $completed = array_filter($fields);
        return intval((count($completed) / count($fields)) * 100);
    }

    public function experiences()
    {
        return $this->hasMany(\App\Models\Experience::class, 'student_id', 'user_id');
    }

    public function educations()
    {
        return $this->hasMany(\App\Models\Education::class, 'student_id', 'user_id');
    }

    public function skills()
    {
        return $this->hasMany(\App\Models\Skill::class, 'student_id', 'user_id');
    }
}
