<?php

namespace App\Http\Controllers;

use App\Models\Skill;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class SkillController extends Controller
{
    // List all skills for a student
    public function index($student_id)
    {
        $this->authorize('viewAny', [Skill::class, $student_id]);
        $skills = Skill::where('student_id', $student_id)->get();
        return response()->json(['skills' => $skills]);
    }

    // Show a single skill
    public function show($id)
    {
        $skill = Skill::findOrFail($id);
        $this->authorize('view', $skill);
        return response()->json($skill);
    }

    // Create a new skill
    public function store(Request $request, $student_id)
    {
        $this->authorize('create', [Skill::class, $student_id]);
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'level' => 'nullable|string|max:255',
        ]);
        $validated['id'] = (string) Str::uuid();
        $validated['student_id'] = $student_id;
        $skill = Skill::create($validated);
        return response()->json($skill, 201);
    }

    // Create multiple skills at once
    public function storeBatch(Request $request, $student_id)
    {
        $this->authorize('create', [Skill::class, $student_id]);
        
        $validated = $request->validate([
            'skills' => 'required|array|max:50', // Limit to 50 skills for safety
            'skills.*.name' => 'required|string|max:255',
            'skills.*.level' => 'nullable|string|max:255',
        ]);

        $skillsData = [];
        foreach ($validated['skills'] as $skillData) {
            $skillsData[] = [
                'id' => (string) Str::uuid(),
                'student_id' => $student_id,
                'name' => $skillData['name'],
                'level' => $skillData['level'] ?? null,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Use batch insert for better performance
        Skill::insert($skillsData);
        
        // Return the created skills
        $createdSkills = Skill::where('student_id', $student_id)
            ->whereIn('name', array_column($validated['skills'], 'name'))
            ->get();

        return response()->json([
            'message' => 'Skills created successfully',
            'count' => count($skillsData),
            'skills' => $createdSkills
        ], 201);
    }

    // Update a skill
    public function update(Request $request, $id)
    {
        $skill = Skill::findOrFail($id);
        $this->authorize('update', $skill);
        $validated = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'level' => 'nullable|string|max:255',
        ]);
        $skill->update($validated);
        return response()->json($skill);
    }

    // Delete a skill
    public function destroy($id)
    {
        $skill = Skill::findOrFail($id);
        $this->authorize('delete', $skill);
        $skill->delete();
        return response()->json(['message' => 'Skill deleted']);
    }
} 