<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Student;
use App\Models\Skill;
use App\Models\Education;

class PublicProfileController extends Controller
{
    public function apiShow($uuid)
    {
        $user = auth()->user();
        
        // Allow access for:
        // 1. Recruiters (can view any active student profile)
        // 2. Students viewing their own profile
        if (!$user) {
            return response()->json(['message' => 'Authentication required.'], 401);
        }
        
        $student = Student::with(['skills', 'educations', 'user'])->where('user_id', $uuid)->first();
        
        if (!$student) {
            return response()->json(['message' => 'Profile not found.'], 404);
        }
        
        // Check authorization
        $isRecruiter = $user->role === 'recruiter';
        $isOwnProfile = $user->role === 'student' && $user->id == $uuid;
        
        if (!$isRecruiter && !$isOwnProfile) {
            return response()->json(['message' => 'Access denied.'], 403);
        }
        
        // For recruiters, check if the profile is active
        if ($isRecruiter && !$student->is_card_active) {
            return response()->json(['message' => 'Profile disabled.'], 403);
        }
        
        // Log activity only for recruiters viewing other profiles
        if ($isRecruiter && !$isOwnProfile) {
            // Log activity for the recruiter (who performed the action)
            \App\Models\ActivityLog::create([
                'user_id' => $user->id,
                'action' => 'viewed_student_profile',
                'details' => "Viewed profile of {$student->full_name} (ID: {$student->user_id})"
            ]);
            
            // Also log activity for the student (whose profile was viewed)
            \App\Models\ActivityLog::create([
                'user_id' => $student->user_id,
                'action' => 'profile_viewed',
                'details' => "Profile viewed by recruiter {$user->name} (ID: {$user->id})"
            ]);
        }
        
        // Get skills and educations from normalized tables
        $skills = $student->skills->map(function($skill) {
            return ['name' => $skill->name, 'level' => $skill->level];
        })->take(5)->toArray();
        
        // Format educations with proper field mapping
        $educations = $student->educations->map(function($education) {
            return [
                'id' => $education->id,
                'institution' => $education->school,
                'degree' => $education->degree,
                'field_of_study' => $education->field,
                'start_date' => $education->start_year,
                'end_date' => $education->end_year,
                'description' => $education->description,
            ];
        })->toArray();
        
        $languages = $student->languages ?? [];
        
        // Convert relative photo URL to full URL if it exists
        $photoUrl = $student->photo_url;
        if ($photoUrl && !str_starts_with($photoUrl, 'http')) {
            $photoUrl = url($photoUrl);
        }
        
        // Return data in format expected by frontend
        return response()->json([
            'student' => [
                'user_id' => $student->user_id,
                'full_name' => $student->full_name,
                'email' => $student->user->email ?? '',
                'photo' => $photoUrl,
                'university' => $student->university,
                'degree' => $student->degree,
                'summary' => $student->summary,
                'linkedin' => $student->linkedin,
                'github' => $student->github,
                'skills' => $skills,
                'educations' => $educations,
                'languages' => $languages,
            ]
        ]);
    }
} 