<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class CvAnalysisController extends Controller
{
    public function analyze(Request $request)
    {
        $request->validate([
            'cv' => 'required|file|mimes:pdf|max:5120',
        ]);

        $path = $request->file('cv')->store('temp');
        $fullPath = storage_path('app/' . $path);

        Log::info('Uploaded file saved at: ' . $fullPath);

        $scriptPath = base_path('extract_cv_text.py');
        $command = escapeshellcmd("python " . escapeshellarg($scriptPath) . " " . escapeshellarg($fullPath)) . " 2>&1";
        $output = shell_exec($command);

        Storage::delete($path);

        Log::info('Command: ' . $command);
        Log::info('Output: ' . $output);

        $apiKey = env('GROQ_API_KEY');
        
        Log::info('Environment GROQ_API_KEY check: ' . ($apiKey ? 'Found (' . strlen($apiKey) . ' chars)' : 'Not found'));
        Log::info('Raw API key value: ' . var_export($apiKey, true));
        
        if (!$apiKey || trim($apiKey) === '') {
            Log::error('GROQ_API_KEY not found in environment variables or is empty');
            return response()->json([
                'structured_cv' => null,
                'llm_raw' => null,
                'extracted_text' => $output,
                'error' => 'AI service not configured properly',
            ], 500);
        }
        
        Log::info('Using Groq API with key: ' . substr($apiKey, 0, 10) . '...');
        $prompt = "Extract the following fields from the provided CV content and return a valid JSON object with exactly these keys:\n\n" .
            "Basic Profile Fields:\n" .
            "- FullName: Full name of the person\n" .
            "- Email: Email address\n" .
            "- Phone: Phone number\n" .
            "- Linkedin: LinkedIn profile URL or username\n" .
            "- Github: GitHub profile URL or username\n" .
            "- Summary: Professional summary or objective (max 500 characters)\n" .
            "- University: Current or most recent university\n" .
            "- GraduationYear: Expected or actual graduation year\n" .
            "- Languages: Array of languages spoken (e.g., [\"English\", \"French\"])\n\n" .
            "Skills Array (for normalized skills table):\n" .
            "- Skills: Array of skill objects with 'name' and 'level' fields. Example:\n" .
            "  [{\"name\": \"JavaScript\", \"level\": \"Advanced\"}, {\"name\": \"Python\", \"level\": \"Intermediate\"}]\n" .
            "  Use levels: Beginner, Intermediate, Advanced, Expert\n" .
            "  IMPORTANT SKILL EXTRACTION GUIDELINES:\n" .
            "  • Group similar technologies under broader categories (e.g., 'JavaScript' instead of 'React, Vue, Angular' separately)\n" .
            "  • Use general skill names (e.g., 'Web Development' instead of 'HTML, CSS, JavaScript' as 3 separate skills)\n" .
            "  • Combine related frameworks/libraries (e.g., 'React' instead of 'React.js, React Hooks, Redux')\n" .
            "  • Focus on major skill categories: Programming Languages, Frameworks, Databases, Cloud Platforms, Tools, Soft Skills\n" .
            "  • Limit to maximum 15 skills total to avoid overwhelming the profile\n" .
            "  • Prioritize the most important/relevant skills mentioned\n" .
            "  • Use standard industry terms (e.g., 'Machine Learning' not 'ML algorithms and data preprocessing')\n" .
            "  • Group version-specific tools (e.g., 'Docker' not 'Docker 20.10, Docker Compose')\n\n" .
            "Education Array (for normalized education table):\n" .
            "- Education: Array of education objects with 'school', 'degree', 'field', 'start_year', 'end_year', 'description' fields. Example:\n" .
            "  [{\"school\": \"University of Technology\", \"degree\": \"Bachelor of Science\", \"field\": \"Computer Science\", \"start_year\": \"2020\", \"end_year\": \"2024\", \"description\": \"Focused on software engineering and web development\"}]\n" .
            "  IMPORTANT EDUCATION EXTRACTION GUIDELINES:\n" .
            "  • Extract ONLY formal academic education from accredited institutions (universities, colleges, schools)\n" .
            "  • INCLUDE: Bachelor's degrees, Master's degrees, PhD, Associate degrees, High School diplomas, formal academic programs\n" .
            "  • EXCLUDE: Professional certifications, online courses, bootcamps, training programs, workshops, seminars, or non-academic credentials\n" .
            "  • For each education entry, identify: institution name, degree type, field of study, duration, and any relevant academic details\n" .
            "  • Use standard academic degree names (e.g., 'Bachelor of Science', 'Master of Arts', 'PhD', 'Associate Degree', 'High School Diploma')\n" .
            "  • For ongoing academic education, leave 'end_year' as empty string or use 'Present'\n" .
            "  • Extract relevant coursework, academic projects, or academic achievements in the description field\n" .
            "  • Standardize field names (e.g., 'Computer Science', 'Business Administration', 'Mechanical Engineering')\n" .
            "  • Include GPA, academic honors, or distinctions in the description if mentioned\n" .
            "  • Do NOT include professional certifications, technical certifications, or any non-academic training as education entries\n\n" .
            "Experience Array (for normalized experience table):\n" .
            "- Experience: Array of experience objects with 'title', 'company', 'start_date', 'end_date', 'description' fields. Example:\n" .
            "  [{\"title\": \"Software Developer\", \"company\": \"Tech Corp\", \"start_date\": \"2023-01-01\", \"end_date\": \"2023-12-31\", \"description\": \"Developed web applications\"}]\n\n" .
            "Instructions:\n" .
            "- If a field is missing, return it as an empty string or empty array as appropriate\n" .
            "- For dates, use YYYY-MM-DD format when possible, or YYYY for years only\n" .
            "- For skills, prioritize broader categories over specific tools/versions\n" .
            "- Infer skill levels based on context: years of experience, project complexity, job titles, self-descriptions\n" .
            "- For education, include ONLY formal academic education from accredited institutions (no certifications, courses, or training)\n" .
            "- For education, extract full institution names and standardize academic degree types and field names\n" .
            "- For education, include relevant academic projects, coursework, or achievements in description\n" .
            "- For experience, include all work experience, internships, and relevant projects\n" .
            "- Consolidate similar skills under one general term (e.g., 'Frontend Development' instead of separate HTML/CSS/JS)\n" .
            "- Focus on transferable, high-level skills rather than specific tool versions\n" .
            "- Ensure education entries are limited to formal academic qualifications only\n" .
            "- Do not include any extra fields\n" .
            "- Return only the JSON object, no markdown or explanation\n\n" .
            "CV Content:\n" . $output;

        $response = Http::withOptions(['verify' => false])->withHeaders([
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $apiKey,
        ])->post('https://api.groq.com/openai/v1/chat/completions', [
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ],
            'model' => 'llama-3.3-70b-versatile',
            'stream' => false,
            'temperature' => 0
        ]);

        Log::info('Groq API response status: ' . $response->status());
        Log::info('Groq API response: ' . $response->body());

        if (!$response->successful()) {
            Log::error('Groq API request failed with status: ' . $response->status());
            return response()->json([
                'structured_cv' => null,
                'llm_raw' => null,
                'extracted_text' => $output,
                'error' => 'AI service unavailable. Status: ' . $response->status(),
            ], 500);
        }

        $json = $response->json();
        Log::info('Parsed JSON response: ' . json_encode($json));

        // Check if the response has the expected structure (Groq uses OpenAI format)
        if (!isset($json['choices']) || !is_array($json['choices']) || empty($json['choices'])) {
            Log::error('Invalid Groq response structure: no choices found');
            return response()->json([
                'structured_cv' => null,
                'llm_raw' => null,
                'extracted_text' => $output,
                'error' => 'Invalid AI response structure',
            ], 500);
        }

        $llm_output = $json['choices'][0]['message']['content'] ?? null;
        Log::info('LLM raw output: ' . ($llm_output ? substr($llm_output, 0, 500) . '...' : 'null'));

        if (!$llm_output) {
            Log::error('No text content in Groq response');
            return response()->json([
                'structured_cv' => null,
                'llm_raw' => null,
                'extracted_text' => $output,
                'error' => 'No content returned from AI service',
            ], 500);
        }

        // Attempt to parse the JSON
        $structured = null;
        if ($llm_output) {
            // Clean up the output - remove markdown formatting if present
            $cleaned_output = $llm_output;
            if (str_starts_with(trim($llm_output), '```json')) {
                $cleaned_output = preg_replace('/^```json\s*/', '', $llm_output);
                $cleaned_output = preg_replace('/\s*```$/', '', $cleaned_output);
            } elseif (str_starts_with(trim($llm_output), '```')) {
                $cleaned_output = preg_replace('/^```\s*/', '', $llm_output);
                $cleaned_output = preg_replace('/\s*```$/', '', $cleaned_output);
            }
            
            Log::info('Cleaned LLM output for parsing: ' . substr($cleaned_output, 0, 500) . '...');
            
            $structured = json_decode(trim($cleaned_output), true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                Log::error('JSON parsing failed: ' . json_last_error_msg());
                Log::error('Raw content that failed to parse: ' . $cleaned_output);
            } else {
                Log::info('Successfully parsed structured data: ' . json_encode($structured));
            }
        }

        return response()->json([
            'structured_cv' => $structured,
            'llm_raw' => $llm_output,
            'extracted_text' => $output,
        ]);
    }
} 