<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Student;
use App\Models\Recruiter;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class AuthController extends Controller
{
    /**
     * Register a new user (student or recruiter)
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => ['required', Rule::in(['student', 'recruiter'])],
            'full_name' => 'required|string|max:255',
            // New student fields
            'phone' => 'nullable|string|max:50',
            'linkedin' => 'nullable|string|max:255',
            'github' => 'nullable|string|max:255',
            'summary' => 'nullable|string',
            'university' => 'nullable|string|max:255',
            'graduation_year' => 'nullable|string|max:50',
            'languages' => 'nullable', // can be string or array
            // Company info for recruiters
            'company_name' => 'required_if:role,recruiter|string|max:255',
            'company_description' => 'nullable|string',
            'company_industry' => 'nullable|string|max:255',
            'company_size' => 'nullable|string|max:50',
            'company_website' => 'nullable|url|max:255',
            // Recruiter position info
            'position' => 'required_if:role,recruiter|string|max:255',
            'department' => 'required_if:role,recruiter|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Create the user
            $user = User::create([
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'role' => $request->role,
            ]);

            // Create role-specific profile
            if ($user->role === 'student') {
                // Parse languages as arrays if needed
                $languages = $request->languages;
                if (is_string($languages)) {
                    $languages = array_map('trim', explode(',', $languages));
                }
                Student::create([
                    'user_id' => $user->id,
                    'full_name' => $request->full_name,
                    'is_card_active' => true,
                    'phone' => $request->phone,
                    'linkedin' => $request->linkedin,
                    'github' => $request->github,
                    'summary' => $request->summary,
                    'university' => $request->university,
                    'graduation_year' => $request->graduation_year,
                    'languages' => $languages,
                ]);
            } elseif ($user->role === 'recruiter') {
                // Create or find company
                $company = Company::firstOrCreate(
                    ['name' => $request->company_name],
                    [
                        'description' => $request->company_description,
                        'industry' => $request->company_industry,
                        'size' => $request->company_size,
                        'website' => $request->company_website,
                    ]
                );

                // If company exists, update its details if new data provided
                if ($request->has('company_description') || $request->has('company_industry') || 
                    $request->has('company_size') || $request->has('company_website')) {
                    $company->update(array_filter([
                        'description' => $request->company_description,
                        'industry' => $request->company_industry,
                        'size' => $request->company_size,
                        'website' => $request->company_website,
                    ]));
                }

                Recruiter::create([
                    'user_id' => $user->id,
                    'company_id' => $company->id,
                    'full_name' => $request->full_name,
                    'position' => $request->position,
                    'department' => $request->department,
                ]);
            }

            // Create token manually for now
            $tokenResult = $user->createToken('PopCareer Token');
            $token = $tokenResult->plainTextToken;

            // Get user with profile
            $userData = $this->getUserWithProfile($user);

            return response()->json([
                'status' => 'success',
                'message' => 'User registered successfully',
                'data' => [
                    'user' => $userData,
                    'token' => $token
                ]
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Registration failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Login user
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        if (!Auth::attempt($request->only('email', 'password'))) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid credentials'
            ], 401);
        }

        $user = Auth::user();
        $tokenResult = $user->createToken('PopCareer Token');
        $token = $tokenResult->plainTextToken;

        // Get user with profile
        $userData = $this->getUserWithProfile($user);

        return response()->json([
            'status' => 'success',
            'message' => 'Login successful',
            'data' => [
                'user' => $userData,
                'token' => $token
            ]
        ]);
    }

    /**
     * Logout user
     */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Logged out successfully'
        ]);
    }

    /**
     * Get authenticated user profile
     */
    public function user(Request $request)
    {
        // Reload the user with relationships
        $user = Auth::user()->fresh(['student.experiences', 'recruiter.company']);

        // Format the user data
        $userData = $this->getUserWithProfile($user);

        return response()->json([
            'status' => 'success',
            'data' => [
                'user' => $userData
            ]
        ]);
    }

    /**
     * Update user profile
     */
    public function updateProfile(Request $request)
    {
        $user = $request->user();
        
        $validator = Validator::make($request->all(), [
            'full_name' => 'sometimes|string|max:255',
            'photo_url' => 'sometimes|string|max:500',
            'cv_file_url' => 'sometimes|string|max:500',
            'company_name' => 'sometimes|string|max:255',
            // Student fields
            'phone' => 'sometimes|string|max:50',
            'linkedin' => 'sometimes|string|max:255',
            'github' => 'sometimes|string|max:255',
            'summary' => 'sometimes|string',
            'university' => 'sometimes|string|max:255',
            'graduation_year' => 'sometimes|string|max:50',
            'languages' => 'sometimes|array',
            // Company fields (for recruiters)
            'company_description' => 'sometimes|string',
            'company_industry' => 'sometimes|string|max:255',
            'company_size' => 'sometimes|string|max:100',
            'company_website' => 'sometimes|string|max:255',
            'company_logo_url' => 'sometimes|string|max:500',
            // Recruiter fields
            'position' => 'sometimes|string|max:255',
            'department' => 'sometimes|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            if ($user->isStudent()) {
                $profile = $user->student;
                if ($profile) {
                    $data = $request->only([
                        'full_name', 'photo_url', 'cv_file_url',
                        'phone', 'linkedin', 'github', 'summary', 'university', 'graduation_year', 'languages'
                    ]);
                    $oldCv = $profile->cv_file_url;
                    $profile->update($data);
                    // Activity log
                    if (isset($data['cv_file_url']) && $data['cv_file_url'] !== $oldCv) {
                        \App\Models\ActivityLog::create([
                            'user_id' => $user->id,
                            'action' => 'Uploaded CV',
                        ]);
                    } else {
                        \App\Models\ActivityLog::create([
                            'user_id' => $user->id,
                            'action' => 'Edited profile',
                        ]);
                    }
                }
            } elseif ($user->isRecruiter()) {
                $profile = $user->recruiter;
                if ($profile) {
                    // Update recruiter profile fields
                    $profile->update($request->only(['full_name', 'position', 'department']));
                    
                    // Update or create company if company fields are provided
                    if ($request->has('company_name')) {
                        $company = Company::firstOrCreate([
                            'name' => $request->company_name
                        ]);
                        
                        // Update company details
                        $companyData = [];
                        if ($request->has('company_description')) {
                            $companyData['description'] = $request->company_description;
                        }
                        if ($request->has('company_industry')) {
                            $companyData['industry'] = $request->company_industry;
                        }
                        if ($request->has('company_size')) {
                            $companyData['size'] = $request->company_size;
                        }
                        if ($request->has('company_website')) {
                            $companyData['website'] = $request->company_website;
                        }
                        if ($request->has('company_logo_url')) {
                            $companyData['logo_url'] = $request->company_logo_url;
                        }
                        
                        if (!empty($companyData)) {
                            $company->update($companyData);
                        }
                        
                        $profile->update(['company_id' => $company->id]);
                    }
                }
            }

            // Get updated user data
            $userData = $this->getUserWithProfile($user->fresh());

            return response()->json([
                'status' => 'success',
                'message' => 'Profile updated successfully',
                'data' => [
                    'user' => $userData
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Profile update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Activate or deactivate the student's card (QR code profile)
     */
    public function setCardActivation(Request $request)
    {
        $user = $request->user();
        if (!$user->isStudent()) {
            return response()->json(['message' => 'Only students can update card activation'], 403);
        }
        $request->validate([
            'is_card_active' => 'required|boolean',
        ]);
        $student = $user->student;
        if (!$student) {
            return response()->json(['message' => 'Student profile not found'], 404);
        }
        $student->is_card_active = $request->is_card_active;
        $student->save();
        return response()->json(['message' => 'Card activation updated', 'is_card_active' => $student->is_card_active]);
    }

    /**
     * Upload profile photo for student
     */
    public function uploadPhoto(Request $request)
    {
        $user = $request->user();
        if (!$user->isStudent()) {
            return response()->json(['message' => 'Only students can upload a photo.'], 403);
        }

        $request->validate([
            'photo' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:5120', // 5MB max
        ]);

        try {
            $file = $request->file('photo');
            if (!$file) {
                return response()->json(['message' => 'No file uploaded'], 400);
            }

            // Generate unique filename
            $filename = 'profile_' . uniqid() . '.' . $file->getClientOriginalExtension();

            // Store using Laravel's Storage facade
            $path = Storage::disk('public')->putFileAs('profile_photos', $file, $filename);

            if (!$path) {
                Log::error('Failed to store file');
                return response()->json(['message' => 'Failed to store file'], 500);
            }

            // Verify file was created
            if (!Storage::disk('public')->exists($path)) {
                Log::error('File was not actually created at: ' . $path);
                return response()->json(['message' => 'File storage verification failed'], 500);
            }

            // Use API route to serve the file (more reliable than symlink on Windows)
            $fullUrl = url('/api/storage/' . $path);

            // Update user
            $user->student->update(['photo_url' => $fullUrl]);

            Log::info('Photo uploaded successfully for user: ' . $user->id . ' at ' . $fullUrl);

            return response()->json(['photo_url' => $fullUrl]);
        } catch (\Exception $e) {
            Log::error('Photo upload failed: ' . $e->getMessage() . ' ' . $e->getTraceAsString());
            return response()->json(['message' => 'Failed to upload photo: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Upload profile photo for recruiter or company logo
     */
    public function uploadRecruiterPhoto(Request $request) 
    {
        $user = $request->user();
        if (!$user->isRecruiter()) {
            return response()->json(['message' => 'Only recruiters can upload photos.'], 403);
        }
        
        $request->validate([
            'image' => 'required|image|max:2048', // 2MB max
            'type' => 'required|in:profile,company' // Specify if uploading profile photo or company logo
        ]);

        $path = '';
        $fullUrl = '';

        if ($request->type === 'company') {
            // Ensure recruiter has a company linked
            if (!$user->recruiter->company) {
                 return response()->json(['message' => 'Recruiter does not have a linked company.'], 400);
            }
            $path = $request->file('image')->store('company_logos', 'public');
            $fullUrl = url('/storage/' . $path);
            $user->recruiter->company->update(['logo_url' => $fullUrl]);
        } else {
             $path = $request->file('image')->store('recruiter_photos', 'public');
             $fullUrl = url('/storage/' . $path);
             $user->recruiter->update(['photo_url' => $fullUrl]);
        }
        
        return response()->json(['url' => $fullUrl, 'type' => $request->type]);
    }

    public function checkEmailExists(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);
        $exists = \App\Models\User::where('email', $request->email)->exists();
        return response()->json(['exists' => $exists]);
    }

    /**
     * Get user with appropriate profile data
     */
    private function getUserWithProfile($user)
    {
        // Ensure relationships are loaded
        $user->loadMissing(['student.experiences', 'student.educations', 'student.skills', 'recruiter.company']);

        $userData = $user->toArray();

        if ($user->isStudent()) {
            $userData['profile'] = $user->student;
            $userData['profile_completion'] = $user->student ? $user->student->getProfileCompletion() : 0;
            $userData['profile']['experiences'] = $user->student->experiences;
            $userData['profile']['educations'] = $user->student->educations;
            $userData['profile']['skills'] = $user->student->skills;
            
            // Convert relative photo URL to full URL if it exists
            if ($userData['profile']['photo_url'] && !str_starts_with($userData['profile']['photo_url'], 'http')) {
                $userData['profile']['photo_url'] = url($userData['profile']['photo_url']);
            }
        } elseif ($user->isRecruiter()) {
            $userData['profile'] = $user->recruiter;
            $userData['profile']['company'] = $user->recruiter->company ?? null;
            // Add shortlist count
            $userData['shortlist_count'] = \App\Models\Shortlist::where('recruiter_id', $user->id)->count();
        }

        return $userData;
    }

    /**
     * Get recent activity logs for the authenticated user
     */
    public function recentActivityLogs(Request $request)
    {
        $user = $request->user();
        $logs = \App\Models\ActivityLog::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->limit(3)
            ->get();
        return response()->json(['logs' => $logs]);
    }

    /**
     * Get dashboard statistics for the authenticated student
     */
    public function dashboardStats(Request $request)
    {
        $user = $request->user();
        
        // Only students can access dashboard stats
        if ($user->role !== 'student') {
            return response()->json(['message' => 'Access denied.'], 403);
        }
        
        $now = new \DateTime();
        $last30Days = (clone $now)->modify('-30 days');
        $last7Days = (clone $now)->modify('-7 days');
        $last60Days = (clone $now)->modify('-60 days');
        $last14Days = (clone $now)->modify('-14 days');
        
        // Get all activity logs for the user in the last 60 days for comparison
        $allActivityLogs = \App\Models\ActivityLog::where('user_id', $user->id)
            ->where('created_at', '>=', $last60Days->format('Y-m-d H:i:s'))
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Current period stats
        $profileViews = $allActivityLogs->filter(function($log) use ($last30Days) {
            $logDate = new \DateTime($log->created_at);
            return $logDate >= $last30Days && 
                   ($log->action === 'profile_viewed' ||
                    strpos(strtolower($log->action), 'profile_view') !== false ||
                    strpos(strtolower($log->action), 'viewed_profile') !== false);
        })->count();
        
        $qrScans = $allActivityLogs->filter(function($log) use ($last7Days) {
            $logDate = new \DateTime($log->created_at);
            return $logDate >= $last7Days && 
                   ($log->action === 'scanned_qr_code' ||
                    strpos(strtolower($log->action), 'qr_scan') !== false ||
                    strpos(strtolower($log->action), 'scan') !== false);
        })->count();
        
        $shortlistAdditions = $allActivityLogs->filter(function($log) use ($last30Days) {
            $logDate = new \DateTime($log->created_at);
            return $logDate >= $last30Days && 
                   (strpos(strtolower($log->action), 'shortlist') !== false ||
                    strpos(strtolower($log->action), 'added_to_shortlist') !== false);
        })->count();
        
        // Previous period stats for comparison
        $previousProfileViews = $allActivityLogs->filter(function($log) use ($last60Days, $last30Days) {
            $logDate = new \DateTime($log->created_at);
            return $logDate >= $last60Days && $logDate < $last30Days &&
                   ($log->action === 'profile_viewed' ||
                    strpos(strtolower($log->action), 'profile_view') !== false ||
                    strpos(strtolower($log->action), 'viewed_profile') !== false);
        })->count();
        
        $previousQrScans = $allActivityLogs->filter(function($log) use ($last14Days, $last7Days) {
            $logDate = new \DateTime($log->created_at);
            return $logDate >= $last14Days && $logDate < $last7Days &&
                   ($log->action === 'scanned_qr_code' ||
                    strpos(strtolower($log->action), 'qr_scan') !== false ||
                    strpos(strtolower($log->action), 'scan') !== false);
        })->count();
        
        $previousShortlistAdditions = $allActivityLogs->filter(function($log) use ($last60Days, $last30Days) {
            $logDate = new \DateTime($log->created_at);
            return $logDate >= $last60Days && $logDate < $last30Days &&
                   (strpos(strtolower($log->action), 'shortlist') !== false ||
                    strpos(strtolower($log->action), 'added_to_shortlist') !== false);
        })->count();
        
        // Calculate percentage changes
        $profileViewsChange = $this->calculatePercentageChange($profileViews, $previousProfileViews);
        $qrScansChange = $this->calculatePercentageChange($qrScans, $previousQrScans);
        $shortlistChange = $this->calculatePercentageChange($shortlistAdditions, $previousShortlistAdditions);
        
        return response()->json([
            'stats' => [
                'profile_views' => $profileViews,
                'profile_views_change' => $profileViewsChange,
                'qr_scans' => $qrScans,
                'qr_scans_change' => $qrScansChange,
                'shortlist_additions' => $shortlistAdditions,
                'shortlist_additions_change' => $shortlistChange,
                'recent_activity' => $allActivityLogs->filter(function($log) use ($last30Days) {
                    $logDate = new \DateTime($log->created_at);
                    return $logDate >= $last30Days;
                })->take(10)->values() // Show last 10 activities from recent period
            ]
        ]);
    }
    
    /**
     * Calculate percentage change between current and previous values
     */
    private function calculatePercentageChange($current, $previous)
    {
        if ($previous == 0) {
            return $current > 0 ? 100 : 0; // If no previous data, show 100% if there's current data
        }
        
        $change = (($current - $previous) / $previous) * 100;
        return round($change, 1);
    }

    /**
     * Log QR code scan activity
     */
    public function logQRScan(Request $request)
    {
        $user = $request->user();
        
        $request->validate([
            'student_id' => 'required|string',
            'student_name' => 'nullable|string'
        ]);
        
        \App\Models\ActivityLog::create([
            'user_id' => $user->id,
            'action' => 'scanned_qr_code',
            'details' => 'Scanned QR code for student: ' . ($request->student_name ?? $request->student_id)
        ]);
        
        return response()->json(['message' => 'QR scan logged successfully']);
    }
}
