<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class AiTipsController extends Controller
{
    public function studentTips(Request $request)
    {
        $user = $request->user()->load([
            'student.skills',
            'student.educations',
            'student.experiences',
        ]);

        if (!$user || !$user->student) {
            return response()->json([
                'status' => 'error',
                'message' => 'Student profile not found',
            ], 404);
        }

        $student = $user->student;
        $profileSnapshot = $this->buildProfileSnapshot($student);
        $apiKey = env('GROQ_API_KEY');

        if (!$apiKey) {
            Log::warning('AI tips request skipped: GROQ_API_KEY missing');
            return $this->fallbackResponse($profileSnapshot, 'AI service not configured');
        }

        $prompt = $this->buildPrompt($profileSnapshot);

        try {
            $response = Http::withOptions(['verify' => false])
                ->withHeaders([
                    'Content-Type' => 'application/json',
                    'Authorization' => 'Bearer ' . $apiKey,
                ])
                ->post('https://api.groq.com/openai/v1/chat/completions', [
                    'messages' => [
                        [
                            'role' => 'user',
                            'content' => $prompt,
                        ],
                    ],
                    'model' => 'llama-3.3-70b-versatile',
                    'temperature' => 0.4,
                    'stream' => false,
                ]);
        } catch (\Throwable $e) {
            Log::error('AI tips request failed: ' . $e->getMessage());
            return $this->fallbackResponse($profileSnapshot, 'AI service unavailable');
        }

        if (!$response->successful()) {
            Log::error('AI tips request failed with status: ' . $response->status());
            return $this->fallbackResponse($profileSnapshot, 'AI request error');
        }

        $json = $response->json();
        $llmOutput = $json['choices'][0]['message']['content'] ?? null;

        if (!$llmOutput) {
            Log::error('AI tips response missing content');
            return $this->fallbackResponse($profileSnapshot, 'Empty AI response');
        }

        $parsedTips = $this->parseTips($llmOutput);
        if (!$parsedTips) {
            Log::error('AI tips JSON parsing failed');
            return $this->fallbackResponse($profileSnapshot, 'Invalid AI response format');
        }

        return response()->json([
            'status' => 'success',
            'source' => 'ai',
            'tips' => $parsedTips,
            'profile_snapshot' => $profileSnapshot,
        ]);
    }

    private function buildProfileSnapshot($student): array
    {
        $skills = $student->skills->map(function ($skill) {
            return [
                'name' => $skill->name,
                'level' => $skill->level,
            ];
        })->values();

        $experiences = $student->experiences->map(function ($exp) {
            return [
                'title' => $exp->title,
                'company' => $exp->company,
                'description' => $exp->description,
                'start_date' => $exp->start_date,
                'end_date' => $exp->end_date,
            ];
        })->values();

        $educations = $student->educations->map(function ($edu) {
            return [
                'school' => $edu->school,
                'degree' => $edu->degree,
                'field' => $edu->field,
                'start_year' => $edu->start_year,
                'end_year' => $edu->end_year,
            ];
        })->values();

        return [
            'full_name' => $student->full_name,
            'summary' => $student->summary,
            'university' => $student->university,
            'graduation_year' => $student->graduation_year,
            'languages' => $student->languages ?? [],
            'linkedin' => $student->linkedin,
            'github' => $student->github,
            'phone' => $student->phone,
            'photo_url' => $student->photo_url,
            'qr_active' => (bool) $student->is_card_active,
            'profile_completion' => $this->calculateProfileCompletion($student, $skills, $experiences, $educations),
            'skills' => $skills,
            'experiences' => $experiences,
            'educations' => $educations,
        ];
    }

    private function buildPrompt(array $snapshot): string
    {
        return "You are a concise career coach for students. Based on the JSON profile snapshot, return 4-6 prioritized, actionable pro tips to improve profile strength, credibility, and recruiter conversions. Each tip must be short and specific to the provided data. Respond with only compact JSON in this exact shape and nothing else:\n\n{\n  \"tips\": [{\n    \"title\": string,\n    \"description\": string,\n    \"action\": string,\n    \"category\": one of [\"profile\", \"skills\", \"experience\", \"branding\", \"readiness\"],\n    \"priority\": one of [\"high\", \"medium\", \"low\"],\n    \"impact_score\": integer 1-5,\n    \"metric\": string // which metric it improves (profile_completion, QR_scans, recruiter_interest)\n  }]\n}\n\nGuidelines:\n- Use the student's gaps and strengths.\n- Prefer measurable, low-effort actions.\n- Avoid generic statements.\n- Keep title <= 8 words and description <= 160 characters.\n- If data is missing, suggest how to fill it.\n\nProfile snapshot JSON:\n" . json_encode($snapshot);
    }

    private function parseTips(string $content): ?array
    {
        $clean = $content;
        if (str_starts_with(trim($clean), '```')) {
            $clean = preg_replace('/^```json?\\s*/', '', $clean);
            $clean = preg_replace('/\s*```$/', '', $clean);
        }

        $decoded = json_decode(trim($clean), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return null;
        }

        $tips = $decoded['tips'] ?? null;
        if (!is_array($tips)) {
            return null;
        }

        return collect($tips)
            ->map(function ($tip) {
                return [
                    'title' => $tip['title'] ?? 'Pro tip',
                    'description' => $tip['description'] ?? '',
                    'action' => $tip['action'] ?? '',
                    'category' => $tip['category'] ?? 'profile',
                    'priority' => $tip['priority'] ?? 'medium',
                    'impact_score' => isset($tip['impact_score']) ? intval($tip['impact_score']) : 3,
                    'metric' => $tip['metric'] ?? 'profile_completion',
                ];
            })
            ->take(6)
            ->values()
            ->toArray();
    }

    private function calculateProfileCompletion($student, $skills, $experiences, $educations): int
    {
        $criteria = [
            !empty($student->full_name),
            !empty($student->summary),
            !empty($student->linkedin),
            !empty($student->github),
            !empty($student->phone),
            !empty($student->photo_url),
            !empty($student->university),
            !empty($student->graduation_year),
            $skills->count() > 0,
            $experiences->count() > 0,
            $educations->count() > 0,
        ];

        $completed = array_filter($criteria);
        $completion = count($criteria) > 0 ? (count($completed) / count($criteria)) * 100 : 0;
        return (int) round($completion);
    }

    private function fallbackResponse(array $snapshot, string $reason)
    {
        $fallbackTips = [
            [
                'title' => 'Add measurable wins',
                'description' => 'Rewrite one experience with metrics (impact, scale, tools) to lift credibility.',
                'action' => 'Update one experience with numbers today.',
                'category' => 'experience',
                'priority' => 'high',
                'impact_score' => 4,
                'metric' => 'recruiter_interest',
            ],
            [
                'title' => 'Strengthen headline',
                'description' => 'Add a sharp 2-line summary with role, focus area, and tech stack.',
                'action' => 'Edit summary on profile.',
                'category' => 'branding',
                'priority' => 'high',
                'impact_score' => 4,
                'metric' => 'profile_completion',
            ],
            [
                'title' => 'Show 5 core skills',
                'description' => 'List your top 5 skills with levels to help recruiters filter you in.',
                'action' => 'Add/update skills with levels.',
                'category' => 'skills',
                'priority' => 'medium',
                'impact_score' => 3,
                'metric' => 'recruiter_interest',
            ],
            [
                'title' => 'Keep QR active',
                'description' => 'Ensure your QR card stays active before events for instant sharing.',
                'action' => 'Toggle card to active.',
                'category' => 'readiness',
                'priority' => 'medium',
                'impact_score' => 3,
                'metric' => 'QR_scans',
            ],
        ];

        return response()->json([
            'status' => 'fallback',
            'source' => 'heuristic',
            'reason' => $reason,
            'tips' => $fallbackTips,
            'profile_snapshot' => $snapshot,
        ]);
    }
}
