# PopCareers Database Schema

## Overview
This document describes the clean, normalized database schema for the PopCareers MVP application. The schema has been optimized to remove redundancies and ensure data consistency.

---

## **Database Tables**

### **1. users**
**Purpose:** Core user authentication and basic information
```sql
CREATE TABLE users (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    email VARCHAR(255) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    role user_role NOT NULL, -- 'student' or 'recruiter'
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### **2. students**
**Purpose:** Student-specific profile information
```sql
CREATE TABLE students (
    user_id UUID PRIMARY KEY REFERENCES users(id) ON DELETE CASCADE,
    full_name VARCHAR(255) NOT NULL,
    phone VARCHAR(50),
    linkedin VARCHAR(255),
    github VARCHAR(255),
    university VARCHAR(255),
    graduation_year VARCHAR(50),
    languages JSONB, -- Array of language objects
    summary TEXT,
    cv_file_url VARCHAR(500),
    qr_code_url VARCHAR(500),
    is_card_active BOOLEAN DEFAULT TRUE,
    photo_url VARCHAR(500)
);
```

### **3. recruiters**
**Purpose:** Recruiter-specific profile information
```sql
CREATE TABLE recruiters (
    user_id UUID PRIMARY KEY REFERENCES users(id) ON DELETE CASCADE,
    company_id UUID REFERENCES companies(id),
    full_name VARCHAR(255) NOT NULL
);
```

### **4. companies**
**Purpose:** Company information for recruiters
```sql
CREATE TABLE companies (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    name VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### **5. experiences**
**Purpose:** Student work experience records (normalized)
```sql
CREATE TABLE experiences (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    student_id UUID REFERENCES students(user_id) ON DELETE CASCADE,
    title VARCHAR(255) NOT NULL,
    company VARCHAR(255),
    start_date DATE,
    end_date DATE,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### **6. educations**
**Purpose:** Student education records (normalized)
```sql
CREATE TABLE educations (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    student_id UUID REFERENCES students(user_id) ON DELETE CASCADE,
    school VARCHAR(255) NOT NULL,
    degree VARCHAR(255),
    field VARCHAR(255),
    start_year VARCHAR(10),
    end_year VARCHAR(10),
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### **7. skills**
**Purpose:** Student skills records (normalized)
```sql
CREATE TABLE skills (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    student_id UUID REFERENCES students(user_id) ON DELETE CASCADE,
    name VARCHAR(255) NOT NULL,
    level VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### **8. shortlists**
**Purpose:** Recruiter shortlists of students
```sql
CREATE TABLE shortlists (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    recruiter_id UUID REFERENCES recruiters(user_id) ON DELETE CASCADE,
    student_id UUID REFERENCES students(user_id) ON DELETE CASCADE,
    private_note TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### **9. activity_logs**
**Purpose:** User activity tracking
```sql
CREATE TABLE activity_logs (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    user_id UUID REFERENCES users(id) ON DELETE CASCADE,
    action VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

---

## **Key Design Decisions**

### **1. Normalization Strategy**
- **Experiences, Educations, Skills**: Moved from JSONB arrays to normalized tables
- **Languages**: Kept as JSONB for flexibility (simple array structure)
- **Removed redundant fields**: `degree` from students (now in educations table)

### **2. UUID Primary Keys**
- All tables use UUIDs for primary keys
- Ensures global uniqueness and security
- Supports distributed systems

### **3. Foreign Key Relationships**
- Proper cascade deletes for data integrity
- Clear relationship mapping between entities

### **4. Data Types**
- **VARCHAR(255)**: Standard for most text fields
- **TEXT**: For longer content (descriptions, summaries)
- **JSONB**: For flexible data structures (languages)
- **DATE**: For date-only fields (experience dates)
- **BOOLEAN**: For simple flags (is_card_active)

---

## **Entity Relationships**

### **User Hierarchy**
```
users (1) ←→ (1) students
users (1) ←→ (1) recruiters
```

### **Student Profile**
```
students (1) ←→ (many) experiences
students (1) ←→ (many) educations
students (1) ←→ (many) skills
```

### **Recruiter Operations**
```
recruiters (1) ←→ (1) companies
recruiters (1) ←→ (many) shortlists
shortlists (many) ←→ (1) students
```

### **Activity Tracking**
```
users (1) ←→ (many) activity_logs
```

---

## **Data Flow Examples**

### **Student Registration**
1. Create `users` record
2. Create `students` record with basic info
3. Later: Add `experiences`, `educations`, `skills` records

### **Recruiter Operations**
1. Create `users` record
2. Create `companies` record (or find existing)
3. Create `recruiters` record
4. Create `shortlists` records when scanning students

### **Profile Management**
- Student updates basic info → `students` table
- Student adds experience → `experiences` table
- Student adds education → `educations` table
- Student adds skill → `skills` table

---

## **Indexes**

### **Performance Indexes**
```sql
-- Students
CREATE INDEX idx_students_is_card_active ON students(is_card_active);

-- Experiences
CREATE INDEX idx_experiences_student_id ON experiences(student_id);

-- Educations
CREATE INDEX idx_educations_student_id ON educations(student_id);

-- Skills
CREATE INDEX idx_skills_student_id ON skills(student_id);

-- Shortlists
CREATE INDEX idx_shortlists_recruiter_id ON shortlists(recruiter_id);
CREATE INDEX idx_shortlists_student_id ON shortlists(student_id);
```

---

## **Migration Notes**

### **Changes Made**
1. **Removed redundant fields**:
   - `skills` JSONB from students (now normalized table)
   - `degree` from students (now in educations table)
   - `profile_data` JSONB (redundant with normalized tables)
   - `public_fields` JSONB (handled by field-level permissions)

2. **Normalized data structures**:
   - Experiences: Separate table with proper relationships
   - Educations: Separate table with degree/field information
   - Skills: Separate table with name/level information

3. **Consistent data types**:
   - All UUIDs for primary keys
   - Proper foreign key constraints
   - Appropriate field lengths

### **Benefits**
- **Data Integrity**: Proper relationships and constraints
- **Performance**: Optimized queries and indexes
- **Scalability**: Normalized structure supports growth
- **Maintainability**: Clear separation of concerns
- **Flexibility**: Easy to extend with new fields

---

## **API Endpoints Mapping**

### **Student Profile CRUD**
- `GET /api/students/{id}/experiences` → `experiences` table
- `POST /api/students/{id}/experiences` → `experiences` table
- `GET /api/students/{id}/educations` → `educations` table
- `POST /api/students/{id}/educations` → `educations` table
- `GET /api/students/{id}/skills` → `skills` table
- `POST /api/students/{id}/skills` → `skills` table

### **Core Profile**
- `GET /api/user` → `users` + `students`/`recruiters` tables
- `PUT /api/profile` → `students` table (basic fields only)

### **Shortlist Management**
- `GET /api/shortlist` → `shortlists` table with joins
- `POST /api/shortlist` → `shortlists` table

---

*Last Updated: Current Session*
*Status: Clean, Normalized Schema* 