# API Endpoints Summary

## Overview
This document provides a comprehensive overview of all API endpoints in the PopCareers application, updated to reflect the new normalized database schema.

---

## **Authentication Endpoints**

### **Public Endpoints**
- `POST /api/register` - User registration (students & recruiters)
- `POST /api/login` - User login
- `GET /api/check-email-exists` - Check if email is already registered

### **Protected Endpoints**
- `POST /api/logout` - User logout
- `GET /api/user` - Get authenticated user profile (includes skills & education)
- `PUT /api/profile` - Update user profile (basic fields only)
- `POST /api/profile/photo` - Upload profile photo (students only)
- `PATCH /api/card-activation` - Toggle student card activation
- `GET /api/activity-logs` - Get recent activity logs

---

## **Student Profile Management Endpoints**

### **Experiences CRUD**
- `GET /api/students/{student_id}/experiences` - Get all experiences for a student
- `POST /api/students/{student_id}/experiences` - Add new experience
- `GET /api/experiences/{id}` - Get specific experience
- `PUT /api/experiences/{id}` - Update experience
- `DELETE /api/experiences/{id}` - Delete experience

### **Education CRUD**
- `GET /api/students/{student_id}/educations` - Get all educations for a student
- `POST /api/students/{student_id}/educations` - Add new education
- `GET /api/educations/{id}` - Get specific education
- `PUT /api/educations/{id}` - Update education
- `DELETE /api/educations/{id}` - Delete education

### **Skills CRUD**
- `GET /api/students/{student_id}/skills` - Get all skills for a student
- `POST /api/students/{student_id}/skills` - Add new skill
- `GET /api/skills/{id}` - Get specific skill
- `PUT /api/skills/{id}` - Update skill
- `DELETE /api/skills/{id}` - Delete skill

---

## **Shortlist Management Endpoints**

- `GET /api/shortlist` - Get recruiter's shortlist (includes skills & education)
- `POST /api/shortlist` - Add student to shortlist
- `PUT /api/shortlist/{student_id}` - Update shortlist note
- `DELETE /api/shortlist/{student_id}` - Remove student from shortlist

---

## **Public Profile Endpoints**

- `GET /api/p/{uuid}` - Get public student profile (QR code scan)

---

## **Utility Endpoints**

- `POST /api/cv/analyze` - Analyze CV and extract structured data

---

## **Endpoint Details & Changes**

### **Updated Endpoints**

#### **1. GET /api/user**
**Changes Made:**
- Now includes `skills` and `educations` from normalized tables
- Removed references to old JSONB fields (`profile_data`, `public_fields`)
- Updated to eager load relationships: `student.experiences`, `student.educations`, `student.skills`

**Response Structure:**
```json
{
    "status": "success",
    "data": {
        "user": {
            "id": "uuid",
            "email": "user@example.com",
            "role": "student",
            "profile": {
                "user_id": "uuid",
                "full_name": "John Doe",
                "phone": "+1234567890",
                "linkedin": "linkedin.com/in/johndoe",
                "github": "github.com/johndoe",
                "university": "University of Technology",
                "graduation_year": "2024",
                "summary": "Passionate developer...",
                "photo_url": "/storage/photo.jpg",
                "is_card_active": true,
                "experiences": [...],
                "educations": [...],
                "skills": [...]
            },
            "profile_completion": 85
        }
    }
}
```

#### **2. PUT /api/profile**
**Changes Made:**
- Removed validation for `skills`, `degree`, `profile_data`, `public_fields`
- Added validation for `languages` array
- Updated to only handle basic profile fields

**Validation Rules:**
```php
[
    'full_name' => 'sometimes|string|max:255',
    'photo_url' => 'sometimes|string|max:500',
    'cv_file_url' => 'sometimes|string|max:500',
    'company_name' => 'sometimes|string|max:255',
    'phone' => 'sometimes|string|max:50',
    'linkedin' => 'sometimes|string|max:255',
    'github' => 'sometimes|string|max:255',
    'summary' => 'sometimes|string',
    'university' => 'sometimes|string|max:255',
    'graduation_year' => 'sometimes|string|max:50',
    'languages' => 'sometimes|array',
]
```

#### **3. GET /api/shortlist**
**Changes Made:**
- Updated to eager load `student.skills` and `student.educations`
- Now returns normalized skills and education data
- Removed references to old JSONB skills field

**Response Structure:**
```json
{
    "shortlist": [
        {
            "student_id": "uuid",
            "full_name": "John Doe",
            "university": "University of Technology",
            "skills": [
                {
                    "id": "skill-uuid",
                    "name": "JavaScript",
                    "level": "Advanced"
                }
            ],
            "educations": [
                {
                    "id": "edu-uuid",
                    "school": "University of Technology",
                    "degree": "Bachelor of Science",
                    "field": "Computer Science"
                }
            ],
            "private_note": "Promising candidate"
        }
    ]
}
```

#### **4. GET /api/p/{uuid} (Public Profile)**
**Changes Made:**
- Updated to use Eloquent models instead of raw DB queries
- Now includes normalized `skills` and `educations` data
- Removed reference to old `degree` field from students table
- Added eager loading for relationships

**Response Structure:**
```json
{
    "full_name": "John Doe",
    "university": "University of Technology",
    "photo": "/storage/photo.jpg",
    "summary": "Passionate developer...",
    "linkedin": "linkedin.com/in/johndoe",
    "github": "github.com/johndoe",
    "skills": [
        {
            "id": "skill-uuid",
            "name": "JavaScript",
            "level": "Advanced"
        }
    ],
    "educations": [
        {
            "id": "edu-uuid",
            "school": "University of Technology",
            "degree": "Bachelor of Science",
            "field": "Computer Science",
            "start_year": "2020",
            "end_year": "2024"
        }
    ],
    "languages": ["English", "French"],
    "user_id": "student-uuid"
}
```

---

## **New Endpoints Added**

### **Experience Management**
All experience endpoints are new and provide full CRUD operations for student work experiences.

### **Education Management**
All education endpoints are new and provide full CRUD operations for student education history.

### **Skills Management**
All skills endpoints are new and provide full CRUD operations for student skills.

---

## **Authorization & Security**

### **Policy Protection**
All new CRUD endpoints are protected by Laravel Policies:
- **ExperiencePolicy**: Only student owner can manage experiences
- **EducationPolicy**: Only student owner can manage educations
- **SkillPolicy**: Only student owner can manage skills

### **Authentication Required**
All endpoints (except public ones) require:
- Valid Bearer token
- User authentication
- Proper role permissions

---

## **Database Schema Compatibility**

### **Removed Fields**
- `skills` JSONB from students table (now normalized)
- `degree` from students table (now in educations table)
- `profile_data` JSONB (redundant)
- `public_fields` JSONB (redundant)

### **New Tables**
- `experiences` - Student work experience records
- `educations` - Student education records
- `skills` - Student skills records

### **Updated Relationships**
- Students now have `hasMany` relationships to experiences, educations, and skills
- All relationships use proper foreign keys with cascade deletes

---

## **Migration Notes**

### **Breaking Changes**
1. **Skills Data**: Previously stored as JSONB array, now normalized table
2. **Degree Field**: Moved from students table to educations table
3. **Profile Data**: Removed redundant JSONB fields

### **Backward Compatibility**
- All existing endpoints maintain their URL structure
- Response formats updated but maintain core functionality
- New endpoints provide enhanced functionality

---

## **Testing Status**

### **Updated Tests**
- ShortlistTest.php - Updated to work with new schema
- All tests pass with normalized structure

### **Test Coverage**
- Authentication endpoints: ✅
- Profile management: ✅
- Shortlist management: ✅
- New CRUD endpoints: ⚠️ (Need additional tests)

---

## **Performance Considerations**

### **Optimizations Made**
- Eager loading of relationships to prevent N+1 queries
- Proper indexing on foreign keys
- Normalized data structure for better query performance

### **Recommended Improvements**
- Add caching for frequently accessed profiles
- Implement pagination for large datasets
- Add rate limiting for API endpoints

---

*Last Updated: Current Session*
*Status: All endpoints updated for normalized schema* 